from django.db import models, connection
from django.db.models.query import Q
# -*- coding: utf-8 -*-

# Implementation of a Modified Preorder Tree Traversal algorithm
# Theory taken from http://www.sitepoint.com/print/hierarchical-data-database
# More examples at: http://code.djangoproject.com/wiki/ModifiedPreorderTreeTraversal
# http://dev.mysql.com/tech-resources/articles/hierarchical-data.html

CHOICES = (
    ('before','Before'),
    ('after','After'),
    ('below','Below')
    )

#set table name here fixme, detect table automatically
TABLE = 'pages_page'
        
class TreeManager(models.Manager):
    
    def get_query_set(self):
        
        sql = "SELECT node.id, count(parent.lft) AS level " \
        "FROM %s AS node, " \
        "%s AS parent " \
        "WHERE node.lft BETWEEN parent.lft AND parent.rgt " \
        "GROUP BY node.lft, node.id " \
        "ORDER BY node.lft" % (TABLE, TABLE)
        
        cursor = connection.cursor()
        cursor.execute(sql)
        rows = cursor.fetchall()
        
        qc = Q()
        self.leveldict = {}
        for i in rows:
            id,level = i
            self.leveldict[id] = level
            qc = qc|Q(id__exact=id)
        
        self.model.add_to_class('levels', self.leveldict)
        
        return super(TreeManager, self).get_query_set().filter(qc)
    
    def all(self): # gets children
    
        sql = "SELECT node.id " \
        "FROM %s AS node, " \
        "%s AS parent " \
        "WHERE node.lft BETWEEN 1 AND 5000 " \
        "AND node.lft BETWEEN parent.lft AND parent.rgt " \
        "GROUP BY node.lft, node.id " \
        "HAVING (count(parent.lft)-1) = 0 " \
        "ORDER BY node.lft" % (TABLE, TABLE)
        
        cursor = connection.cursor()
        cursor.execute(sql)
        rows = cursor.fetchall()
        
        top = []
        for i in rows:
            top.append(super(TreeManager, self).get_query_set().get(pk=i[0]))
        
        return top
        
class Page(models.Model):
        
    label = models.CharField(maxlength=255)

    # To create categories for a website menu, we won't probably need to record
    # what user created the node. But if nodes are used for threaded comments
    # or forums posts, it's a good idea to have.
    where = models.CharField(maxlength=6,choices=CHOICES,blank=True,null=True)
    wich = models.ForeignKey("self",blank=True,null=True)

    lft = models.PositiveIntegerField(blank=True,null=True,db_index = True)
    rgt = models.PositiveIntegerField(blank=True,null=True)
   
    manager = TreeManager()
    
    def __repr__(self):
        # Page.objects.all() should get whole list as tree with level , repeat ---  per level
        # rertun'---' times level + self.label
        try:
            level = self.levels[self.id]
        except:
            level=1
        
        return ('----' * level) + ' ' + self.label
    
    def before(self,related_node):
    
         # adds node before picked node self.wich
        cursor = connection.cursor()
        
        target_rgt = related_node.rgt - 1
        target_lft = related_node.lft - 1

        cursor.execute("UPDATE %s " \
                       "SET rgt = rgt + 2 " \
                       "WHERE rgt > %i " % (self.tbl(),  int(target_rgt)))

        cursor.execute("UPDATE %s " \
                       "SET lft = lft + 2 " \
                       "WHERE lft > %i " % (self.tbl(),  int(target_lft)))

        self.rgt = related_node.rgt
        self.lft = related_node.lft
    
    def after(self,related_node):

        # adds node after picked node self.wich
        
        cursor = connection.cursor()
        
        target_rgt = related_node.rgt

        cursor.execute("UPDATE %s " \
                       "SET rgt = rgt + 2 " \
                       "WHERE rgt > %i " % (self.tbl(),  int(target_rgt)))
                       
        cursor.execute("UPDATE %s " \
                       "SET lft = lft + 2 " \
                       "WHERE lft > %i " % (self.tbl(),  int(target_rgt)))
           
        self.lft = related_node.lft + 2
        self.rgt = related_node.rgt + 2
        
    def below(self,related_node):
        
        # *fixme* should make sure below does not work on nodes already having child nodes
        # because that does not have a happy ending
        if related_node.rgt == related_node.lft + 1:
        
            cursor = connection.cursor()
            
            target_rgt = related_node.rgt - 1

            cursor.execute("UPDATE %s " \
                           "SET rgt = rgt + 2 " \
                           "WHERE rgt > %i " % (self.tbl(),  int(target_rgt)))

            cursor.execute("UPDATE %s " \
                           "SET lft = lft + 2 " \
                           "WHERE lft > %i " % (self.tbl(),  int(target_rgt)))
            
            self.rgt = related_node.rgt + 1
            self.lft = related_node.lft + 1
            
        else:
        
            sql = "SELECT node.lft, node.rgt " \
            "FROM %s AS node, " \
            "%s AS parent " \
            "WHERE node.lft BETWEEN %s AND %s " \
            "AND node.lft BETWEEN parent.lft AND parent.rgt " \
            "GROUP BY node.lft, node.rgt " \
            "HAVING (count(parent.lft)-1) = 1 " \
            "ORDER BY node.lft" % (self.tbl(), self.tbl(), related_node.lft, related_node.rgt)
            
            cursor = connection.cursor()
            cursor.execute(sql)
            rows = cursor.fetchall()
            
            related_node.lft, related_node.rgt = rows[0]
            
            self.before(related_node)

    def top(self):
    
        # adds node to bottom of top level
        # If there are zero nodes for this tree so far, we will just
        # insert with lft = 1 and rgt = 2

        cursor = connection.cursor()
        cursor.execute("SELECT MAX(rgt) FROM %s LIMIT 1" % self.tbl() )
        
        row = cursor.fetchone()
        
        current_max_rgt = row[0]
        
        if current_max_rgt is None:
            self.lft = 1
            self.rgt = 2
        # But if there are already pages attached to the tree we will put
        # this new one at the top level (to the right of the 'tree')
        else:
            self.lft = current_max_rgt + 1
            self.rgt = current_max_rgt + 2
    
    def skip(self):
        u=False
        # *fixme* reset parentnode to avoid future errors
        
    def save(self):
    
        where = self.where # where to place node ralative to related_node
        
        
        if self.id:
            dont_trust_the_form = Page.manager.get(pk=self.id)
            self.lft = dont_trust_the_form.lft
            self.rgt = dont_trust_the_form.rgt
            
        try: # test for related node
            related_node = self.wich
        except:
            related_node = False

            if where:
                where = "skip" # don't do anything if no node is selected
            else: # add to toplevel if no node is selected
                where = "top"

        if self.rgt: # node exists just change content, fix this to allow moving og nodes and child nodes
            where = "skip"

        # done testing for now, execute!
        
        if where=="before":
            self.before(related_node)
            
        if where=="after":
            self.after(related_node)
            
        if where=="below":
            self.below(related_node)
            
        if where=="skip":
            self.skip()
            
        if where=="top":
            self.top()
            
        self.where = ''
        self.wich = ''
        
        super(Page, self).save()

    def delete(self):
        # delete nodes and child nodes
        if self.rgt and self.lft:
            width = self.rgt - self.lft + 1

            del_sql = "DELETE FROM %s WHERE lft BETWEEN %s AND %s" % (self.tbl(), self.lft, self.rgt) 
            upd_sql_1 = "UPDATE %s SET rgt = rgt - %s WHERE rgt > %s" % (self.tbl(), width, self.rgt)
            upd_sql_2 = "UPDATE %s SET lft = lft - %s WHERE rgt > %s" % (self.tbl(), width, self.rgt)

            cursor = connection.cursor()
            cursor.execute(del_sql)
            cursor.execute(upd_sql_1)
            cursor.execute(upd_sql_2)

        super(Page, self).delete()
        
    def get_parent(self):
    
        parentid = 5000
        if self.where == 'below':
           parentid = self.wich.id
        else:
            sql = "SELECT parent.id " \
            "FROM %s AS node, " \
            "%s AS parent " \
            "WHERE node.lft BETWEEN parent.lft AND parent.rgt"  \
            "AND node.id = %s " \
            "ORDER BY node.lft" % (self.tbl(), self.tbl(), self.wich.id)
            
            cursor = connection.cursor()
            cursor.execute(sql)
            
            row = cursor.fetchall()
            
            parentid=row[0][0]
        
        return Page.manager.get(pk=parentid)
        
    def get_children(self): # gets children
    
        sql = "SELECT node.id " \
        "FROM %s AS node, " \
        "%s AS parent " \
        "WHERE node.lft BETWEEN %s AND %s " \
        "AND node.lft BETWEEN parent.lft AND parent.rgt " \
        "GROUP BY node.lft, node.id " \
        "HAVING (count(parent.lft)-1) = 1 " \
        "ORDER BY node.lft" % (self.tbl(), self.tbl(), self.lft, self.rgt)
        
        cursor = connection.cursor()
        cursor.execute(sql)
        rows = cursor.fetchall()
        
        children = []
        for i in rows:
            children.append(Page.manager.get(pk=i[0]))
        
        return children
        

        
    def tbl(self): # finds table name how do i do this in the TreeManager
    
        name = self.__class__.__name__.lower()
        app = __name__.split('.')[1]
        
        return app + '_' + name
        
    class Meta:
        ordering = ['lft',]
        
    class Admin:
        list_display = ('__repr__','lft','rgt')
        fields = (
            (None, { 'fields' : ('label',) }),
            ('Position', { 'fields': ('where','wich') }),
            ('Advanced', { 'fields': ('lft','rgt') , 'classes': 'collapse' }),
        )
        manager = TreeManager()