from django.template import Library, Node, TemplateSyntaxError
from django.template import VariableNode, FILTER_SEPARATOR
from django.utils.html import escape

register = Library()

FINALIZED_FILTER_NAME = 'finalized'

class FinalFilterNode(Node):
    def __init__(self, nodelist, filters):
        self.nodelist = nodelist
        self.filters = filters
        self.parsed = False

    def render(self, context):
        self.parse_nodes()
        return self.nodelist.render(context)
    
    def parse_nodes(self):
        if self.parsed:
            # Don't ever parse twice.
            return
        nodelist = self.nodelist
        # Parse inner FinalFilterNodes first (they may contain a
        # finalize filter).
        for node in nodelist.get_nodes_by_type(FinalFilterNode):
            node.parse_nodes()
        # Parse child nodes.
        for node in nodelist.get_nodes_by_type(VariableNode):
            node_filters = node.filter_expression.filters
            filter_funcs = [filter[0] for filter in node_filters]
            if finalized not in filter_funcs:
                # Ignore the node if it has the finalized functions in
                # its filters.
                for filter in self.filters:
                    if filter[0] not in filter_funcs:
                        # Don't double-up on filters which have already
                        # been applied to this VariableNode.
                        node_filters.append(filter)
        self.parsed = True

def finalfilter(parser, token):
    """
    Add common filters to all variable nodes within this block tag.
    Use the `finalized` filter in a variable node to skip adding the filters
    to that node. If a common filter has already been explicitly added to a
    variable node, it will *not* be added again.

    Filters can also be piped through each other, and they can have
    arguments -- just like in variable syntax.

    Note: This tag adds the filter(s) at render time so this happens across
    all extended block tags.

    Example usage::

        {% finalfilter escape %}
            {{ html_menu|finalized }}
            <h2>{{ object.company }}</h2>
            <p>
                <a href="{{ object.url }}">
                    {{ object.first_name }} {{ object.last_name }}
                </a>
            </p>
        {% endfinalfilter %}

    This example would add the escape filter to the end of each variable node
    except for `html_menu`.
    """
    nodelist = parser.parse(('endfinalfilter',))
    parser.delete_first_token()
    
    try:
        _, rest = token.contents.split(None, 1)
    except ValueError:
        raise TemplateSyntaxError, "'finalfilter' requires at least one filter"
    
    # Build the final filters list.
    filter_expression = parser.compile_filter('var%s%s' % (FILTER_SEPARATOR, rest))
    filters = filter_expression.filters

    return FinalFilterNode(nodelist, filters)
finalfilter = register.tag(finalfilter)

def finalized(value):
    """
    Used to cancel the effect of {% finalfilter %}. Has no other effect.
    """
    return value
register.filter(FINALIZED_FILTER_NAME, finalized)

def better_escape(value):
    """
    HTML escape the given text with ampersands, quotes and carets encoded.
    Alternately, if a list is given, each item of the list is html escaped
    (lists are escaped recursively).
    
    Useful for this sort of thing::
    
        {{ names_list|escape|join:'<br />' }}
    """
    if isinstance(value, (list, tuple)):
        return map(better_escape, value)
    else:
        return escape(value)
# It'd be nice to override the default escape filter, but that causes
# problems with double-escaping when escape is used in templates which
# use {% extends %} without {% load filtertags %}.
# Best solution is if this is implemented in core :)
register.filter('better_escape', better_escape)