"""
To demonstrate Django bug #9795:

1. Install this app and the admin.

2. Add some ``Person`` objects, of both genders.

3. In the admin, attempt to create a ``Child`` object and click the
   related-object popup to select the ``Person``.

4. Notice that the ``limit_choices_to`` from ``Mother`` is being
   applied within the related-object popup, even though you're not
   selecting an object to relate to a ``Mother``.

This only happens when the relation between ``Mother`` and ``Person``
is a ``OneToOneField`` with ``limit_choices_to``, and when the
``ModelAdmin`` specified for ``Child`` declares ``person`` in
``raw_id_fields``.

"""


from django.db import models


GENDER_CHOICES = (
    ('m', 'Male'),
    ('f', 'Female'),
    )


class Person(models.Model):
    name = models.CharField(max_length=255)
    gender = models.CharField(max_length=1, choices=GENDER_CHOICES)

    class Meta:
        verbose_name_plural = 'people'

    def __unicode__(self):
        return self.name


class Mother(models.Model):
    """
    Since a mother is, by definition, female, we set
    ``limit_choices_to`` on the relation to ``Person``.

    Making that relation a ``OneToOneField`` exposes the bug.
    
    """
    num_children = models.IntegerField()
    person = models.OneToOneField(Person, limit_choices_to={ 'gender': 'f' })

    def __unicode__(self):
        return "%s: mother of %s" % (self.person, self.num_children)


class Child(models.Model):
    person = models.ForeignKey(Person, unique=True)
    birth_date = models.DateField()

    class Meta:
        verbose_name_plural = 'children'

    def __unicode__(self):
        return "%s born %s" % (self.person, self.birth_date)
