# -*- coding: utf-8 -*-
# vim:enc=utf8:fenc=utf8:ts=4:sts=4:sw=4:sta:et:nofen:si:ai:nowrap
# (C) Djordjevic Nebojsa <nesh@studioquattro.co.yu> 2005
# You're welcome to redistribute this software under the
# terms of the GNU Library General Public Licence version 2.0
# or, at your option, any higher version.
#
# You can read the complete GNU LGPL in the file COPYING
# which should come along with this software, or visit
# the Free Software Foundation's WEB site http://www.fsf.org
#

""" image related filters """

import os, struct
from django.utils.translation import gettext
from django.conf.settings import DEBUG

def _path(src):
    from django.conf.settings import MEDIA_ROOT, MEDIA_URL

    if src.startswith(MEDIA_URL):
        return os.path.join(MEDIA_ROOT, src[len(MEDIA_URL):])
    else:
        return os.path.join(MEDIA_ROOT, src)
#
try:
    import Image
    HAS_PIL = True
except ImportError:
    HAS_PIL = False

def _no_pil_image_size(fname):
    """
    Determine the image type of FNAME and return its size.
    from draco
    """
    try:
        filehandle = file(fname, 'rb')
    except IOError:
        return None
    head = filehandle.read(24)
    if len(head) != 24:
        return
    if head[:4] == '\x89PNG':
        # PNG
        check = struct.unpack('>i', head[4:8])[0]
        if check != 0x0d0a1a0a:
            return
        width, height = struct.unpack('>ii', head[16:24])
    elif head[:6] in ('GIF87a', 'GIF89a'):
        # GIF
        width, height = struct.unpack('<HH', head[6:10])
    elif head[:4] == '\xff\xd8\xff\xe0' and head[6:10] == 'JFIF':
        # JPEG
        try:
            filehandle.seek(0)  # Read 0xff next
            size = 2
            filetype = 0
            while not 0xc0 <= filetype <= 0xcf:
                filehandle.seek(size, 1)
                byte = filehandle.read(1)
                while ord(byte) == 0xff:
                    byte = filehandle.read(1)
                filetype = ord(byte)
                size = struct.unpack('>H', filehandle.read(2))[0] - 2
            # We are at a SOFn block
            filehandle.seek(1, 1)  # Skip `precision' byte.
            height, width = struct.unpack('>HH', filehandle.read(4))
        except:
            raise
            return
    else:
        return
    return width, height
#

from contrib.utils import set_cached_file, get_cached_file

def _image_size(path):
    size = get_cached_file(path)
    if size is None:
        if HAS_PIL:
            size = Image.open(path).size
        else:
            size = _no_pil_image_size(path)
        set_cached_file(path, size)
    #
    return size
#

_THUMB_DIR = 'thumb'

def thumbnail_file(filename):
    from django.conf.settings import MEDIA_ROOT
    basedir = os.path.dirname(filename)
    name = os.path.basename(filename)
    fname, ext = os.path.splitext(name)
    ret = os.path.join(MEDIA_ROOT, basedir, _THUMB_DIR, fname + '.jpg')
    if os.path.isfile(ret):
        return ret
    return None
# thumbnail_file

def remove_thumbnail(field):
    th = thumbnail_file(field)
    if th:
        os.unlink(th)
#

def _make_thumb(path, width, height):
    if not HAS_PIL:
        return path
        
    from django.conf.settings import MEDIA_ROOT
    url = os.path.dirname(path)
    path = _path(path)
    # thumb dir
    basedir = os.path.dirname(path)
    thdir = os.path.join(MEDIA_ROOT, basedir, _THUMB_DIR)

    # create dir if not exists
    if not os.path.isdir(thdir):
        os.mkdir(thdir)

    name = os.path.basename(path)
    fname, ext = os.path.splitext(name)
    outfile = os.path.join(thdir, fname + '.jpg')

    if os.path.isfile(outfile):
        # thumbnail already exists
        return '%s/thumb/%s.jpg' % (url, fname)

    # make and save thumbnail
    try:
        im = Image.open(path)
        w, h = im.size
        if (h != height) or (w != width):
            im.thumbnail((width, height), Image.ANTIALIAS)
            im.save(outfile, "JPEG")
        else:
            return path
    except:
        return path

    return '%s/thumb/%s.jpg' % (url, fname)
# _make_thumb

##################################################
## FILTER ##

from django.core.template import TemplateSyntaxError, TokenParser, Library 
register = Library()

@register.filter()
def img(value, args=''):
    """ Create ``<img>`` tag with all required parameters.

If image dimension is set and dimension is different from original
image size and *thumbnail*\ [#]_ option is not ``False``
it will make required **thumbnail**\ [#]_ and return link to it.

.. [#] default is ``True``.
.. [#] requires PIL_, if PIL_ is not found fails silently.
.. _PIL: http://www.pythonware.com/products/pil/
"""
    
    kwargs = {}
    if args:
        for arg in args.split('|'):
            arg = arg.strip()
            if arg == '': continue
            kw, val = arg.split('=', 1)
            kw = kw.lower()
            try:
                val = int(val) # convert all ints
            except ValueError:
                pass
            kwargs[kw] = val
        # for
    # if

    path = _path(value)

    # file exists?
    if not value:
        if DEBUG:
            return '<div class="error"><p class="title">%s</p></div>' % gettext('no image')
        else:
            return ''
    elif not os.path.isfile(path):
        if DEBUG:
            return '<div class="error"><p class="title">%s</p><p>%s</p></div>' \
                % (gettext('no image'), gettext('image %s not found') % value)
        else:
            return ''

    # no alt, use value
    if 'alt' not in kwargs:
        kwargs['alt'] = value
    #

    if kwargs.get('thumbnail', True) and HAS_PIL:
        if ('width' in kwargs) or ('height' in kwargs):
            size = _image_size(path)
            if size is not None:
                width, height = size
            if 'width' in kwargs:
                new_width = kwargs['width']
                del kwargs['width']
            else:
                new_width = width
            #
            if 'height' in kwargs:
                new_height = kwargs['height']
                del kwargs['height']
            else:
                new_height = height
            #
            if (size is not None) and ((new_width != width) or (new_height != height)):
                value = _make_thumb(value, new_width, new_height)
                path = _path(value)
            # if
        # if
    # if

    size = _image_size(path)
    if size is not None:
        w, h = size
        kwargs['width'] = '%d' % w
        kwargs['height'] = '%d' % h
    #

    ret = ['src="%s"' % value] # TODO: escape
    for k, v in kwargs.items():
        ret.append('%s="%s"' % (k, v))
    return '<img %s />' % ' '.join(ret)
#