"""add(), create() etc. support for Django's intermediate ManyToMany models

This module implements at least part of
http://code.djangoproject.com/ticket/9475 without patching Django.

It only detects which fields have default values, and doesn't try to
be super smart beyond that.

Usage: Instead of models.ManyToManyField, use
       smart_m2mfield.SmartManyToManyField.
"""

from django.utils.functional import curry
from django.db.models.fields import AutoField, NOT_PROVIDED
from django.db.models.fields.related import (
    ReverseManyRelatedObjectsDescriptor, ManyToManyField,
    create_many_to_many_intermediary_model, add_lazy_relation,
    create_many_related_manager)


def has_defaults(field):
    """Return False for through m2m with no defaults

    If the many-to-many relation defined by the given field uses a
    manually defined intermediary model, and the model doesn't define
    defaults for all extra fields, this function returns False.
    """
    for through_field in field.rel.through._meta.fields:
        if (not isinstance(through_field, AutoField)
            and through_field.name not in (field.m2m_field_name(),
                                           field.m2m_reverse_field_name())
            and through_field.default == NOT_PROVIDED):
            return False
    return True


class SmartReverseManyRelatedObjectsDescriptor(
    ReverseManyRelatedObjectsDescriptor):
    """Smarter reverse m2m related objects descriptor

    This is a modified version of ReverseManyRelatedObjectsDescriptor
    to support add(), create() etc. when ``through=`` is used but all
    extra fields of the intermediary model have a default value
    """

    def __get__(self, instance, instance_type=None):
        if instance is None:
            return self

        # Dynamically create a class that subclasses the related
        # model's default manager.
        rel_model = self.field.rel.to
        superclass = rel_model._default_manager.__class__
        RelatedManager = create_many_related_manager(
            superclass, self.field.rel)

        if (not self.field.rel.through._meta.auto_created and
            has_defaults(self.field)):

            def add(self, *objs):
                self._add_items(
                    self.source_field_name, self.target_field_name, *objs)
                if self.symmetrical:
                    self._add_items(
                        self.target_field_name, self.source_field_name, *objs)
            add.alters_data = True

            def remove(self, *objs):
                self._remove_items(
                    self.source_field_name, self.target_field_name, *objs)
                if self.symmetrical:
                    self._remove_items(
                        self.target_field_name, self.source_field_name, *objs)
            remove.alters_data = True

            RelatedManager.add = add
            RelatedManager.remove = remove

        manager = RelatedManager(
            model=rel_model,
            core_filters={'%s__pk' % self.field.related_query_name():
                          instance._get_pk_val()},
            instance=instance,
            symmetrical=(self.field.rel.symmetrical and
                         isinstance(instance, rel_model)),
            source_field_name=self.field.m2m_field_name(),
            target_field_name=self.field.m2m_reverse_field_name(),
            reverse=False)

        return manager

    def __set__(self, instance, value):
        if instance is None:
            raise AttributeError("Manager must be accessed via instance")

        if (not self.field.rel.through._meta.auto_created and
            not has_defaults(self.field)):
            opts = self.field.rel.through._meta
            raise AttributeError(
                "Cannot set values on a ManyToManyField which specifies an "
                "intermediary model with fields which have no default value.  "
                "Use %s.%s's Manager instead." % (
                    opts.app_label, opts.object_name))

        manager = self.__get__(instance)
        manager.clear()
        manager.add(*value)


class SmartManyToManyField(ManyToManyField):

    def contribute_to_class(self, cls, name):
        """Modified version of ManyToManyField.contribute_to_class

        Support support add(), create() etc. when ``through=`` is used
        but all extra fields of the intermediary model have a default
        value.
        """

        # To support multiple relations to self, it's useful to have a non-None
        # related name on symmetrical relations for internal reasons. The
        # concept doesn't make a lot of sense externally ("you want me to
        # specify *what* on my non-reversible relation?!"), so we set it up
        # automatically. The funky name reduces the chance of an accidental
        # clash.
        if self.rel.symmetrical and (
            self.rel.to == "self" or self.rel.to == cls._meta.object_name):
            self.rel.related_name = "%s_rel_+" % name

        super(SmartManyToManyField, self).contribute_to_class(cls, name)

        # The intermediate m2m model is not auto created if:
        #  1) There is a manually specified intermediate, or
        #  2) The class owning the m2m field is abstract.
        if not self.rel.through and not cls._meta.abstract:
            self.rel.through = create_many_to_many_intermediary_model(
                self, cls)

        # Add the descriptor for the m2m relation
        setattr(cls, self.name, SmartReverseManyRelatedObjectsDescriptor(self))

        # Set up the accessor for the m2m table name for the relation
        self.m2m_db_table = curry(self._get_m2m_db_table, cls._meta)

        # Populate some necessary rel arguments so that cross-app relations
        # work correctly.
        if isinstance(self.rel.through, basestring):

            def resolve_through_model(field, model, cls):
                field.rel.through = model

            add_lazy_relation(cls, self, self.rel.through,
                              resolve_through_model)

        if isinstance(self.rel.to, basestring):
            target = self.rel.to
        else:
            target = self.rel.to._meta.db_table
        cls._meta.duplicate_targets[self.column] = (target, "m2m")
