from django.core import meta

class Food(meta.Model):
    parent = meta.ForeignKey('self', blank=True, null=True, related_name='child')
    name   = meta.CharField(maxlength=50)

    class META:
        ordering = ('name',)
        admin = meta.Admin(
            fields = (
                (None, {'fields': ('name', 'parent',)}),
            ),
            list_display  = ('name',),
            search_fields = ['name',],
        )
    
    def __repr__(self):
        return self.name
        
    def get_all_children(self):
        output = []
        children = self.get_child_list()
        for c in children:
            output.append(c)
            output = output + c.get_all_children()
        return output
        
    def get_all_parents(self):
        id_list = []
        object = self
        while True:
            id_list.append(object)
            if not object.parent:
                break
            object = object.get_parent()
        id_list.reverse()
        return id_list
    
class Restaurant(meta.Model):
    foods  = meta.ManyToManyField(Food, filter_interface=meta.HORIZONTAL)
    last_modified = meta.DateTimeField(auto_now=True )
    name   = meta.CharField(maxlength=50)
    lat    = meta.FloatField(max_digits=10, decimal_places=6)
    lng    = meta.FloatField(max_digits=10, decimal_places=6)
    street = meta.CharField(maxlength=50)
    apt    = meta.CharField( maxlength=10, blank=True)
    city   = meta.CharField(maxlength=32)
    state  = meta.USStateField(default='TX')
    zip    = meta.PositiveIntegerField()
    phone  = meta.PhoneNumberField(blank=True)
    image  = meta.ImageField(upload_to='lunchbox/restaurants/images', blank=True, null=True)
    text   = meta.TextField(blank=True, null=True)
    notes  = meta.TextField(blank=True, null=True)

    class META:
        admin = meta.Admin(
            fields = (
                (None,        {'fields': ('name', 'foods', 'phone', 'image',)}),
                ('Geocoding', {'fields': ('lat', 'lng',)}),
                ('Address',   {'fields': ('street', 'apt', 'city', 'state', 'zip',)}),
                (None,        {'fields': ('text', 'notes',)}),
            ),
            list_display  = ('name', 'street', 'city', 'state', 'zip', 'has_image',),
            list_filter   = ('city', 'state', 'zip',),
            ordering      = ('name',),
            save_on_top   = True,
            search_fields = ['name', 'street', 'city',],
            js = (
                'http://localhost/tiny_mce/tiny_mce.js',
                'http://localhost/appmedia/admin/js/textareas.js',
            ),
        )
        get_latest_by = 'last_modified'
    
    def __repr__(self):
        return self.name

    def has_image(self):
        if self.get_image_url():
            return True
        return False
    has_image.short_description = 'image?'

class Link(meta.Model):
    restaurant = meta.ForeignKey(Restaurant, edit_inline=meta.TABULAR, num_in_admin=2)
    url  = meta.URLField (verify_exists=False, core=True)
    name = meta.CharField(maxlength=10, blank=True)
