## Brickit -- a repository and tools for synthetic biology
##     Copyright (C) 2007  Raik Gruenberg

##     Brickit is free software: you can redistribute it and/or modify
##     it under the terms of the GNU General Public License as published by
##     the Free Software Foundation, either version 3 of the License, or
##     (at your option) any later version.

##     Brickit is distributed in the hope that it will be useful,
##     but WITHOUT ANY WARRANTY; without even the implied warranty of
##     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##     GNU General Public License for more details.

##     You should have received a copy of the GNU General Public License
##     along with this program.  If not, see <http://www.gnu.org/licenses/>.


from django.db import models

from django.contrib import databrowse

# Create your models here.


class Reference( models.Model ):
    """
    Minimalistic description of a URL link
    """

    REFERENCE_TYPE_CHOICES = ( ('web_site', 'web site'),
			       ('article', 'article'),
			       ('book', 'book'),
			       ('database_entry', 'database entry'),
			       ('mit_part', 'MIT part') )
    
    name = models.CharField( maxlength=100, primary_key=True )

    #: voluntary brief description
    short_description = models.CharField( maxlength=200, blank=True )

    #: obligate Link
    url = models.URLField()

    #: obligate category
    reference_type = models.CharField( 'type of reference', maxlength=50,
				       choices=REFERENCE_TYPE_CHOICES,
				       default='web site' )
    def __unicode__( self ):
	return self.reference_type + ': ' + self.name

    class Admin:
	pass


class StorageContainer( models.Model ):
    """
    A container holding several physical samples of DNA or other stuff.
    """

    STORAGE_CONTAINER_TYPES= (
	('96-well-plate', '96 well plate'),
	('384-well-plate','384 well plate'),
	('box', 'freezer box'),
	('other', 'other' ) )

    #: bar-code label for this container
    label = models.CharField('(bar code) label', maxlength=15,
			     primary_key=True )

    #: human-readable name
    nickname = models.CharField(maxlength=100)

    #: [optional] annotation
    description = models.CharField(maxlength=200, blank=True)

    #: [optional]
    comments = models.TextField(blank=True)

    #: what type of container is it
    container_type = models.CharField('type of container', maxlength=30,
				      choices=STORAGE_CONTAINER_TYPES )

    location = models.CharField('location', maxlength=200)

    # [optional] links_html
    references = models.ManyToManyField( Reference, blank=True )

    def __unicode__( self ):
	return self.label

    class Admin:
	pass



class Sample( models.Model ):
    """
    Sample describes a single tupe or well holding a bit of DNA or
    another sample (?).
    """

    STORAGE_WELL_TYPES = ( ('tube','tube'), ('well','well'), ('other','other'))
    STORAGE_TYPES      = ( ('dna', 'DNA'), ('cells','cells') )


    label = models.CharField('(bar code) label', maxlength=15,
			     primary_key=True)

    #: link to a single container
    container = models.ForeignKey( StorageContainer )
    
    well_type = models.CharField('type of well or tube', maxlength=30,
				 choices=STORAGE_WELL_TYPES )

    storage_type = models.CharField('stored as', maxlength=100,
				    choices=STORAGE_TYPES )

    #: [optional]
    comments = models.TextField(blank=True)

    #: [optional] (for now)
    well = models.CharField(maxlength=5, blank=True)

    created = models.DateField('created at', auto_now_add=True)

    #: link to the physical DNA contained in this sample
    dna = models.ForeignKey( 'DNA',
			     verbose_name='physical DNA',
			     related_name='samples',
			     blank=False)
    
    def __unicode__( self ):
	return self.container.__unicode__() + ' / ' + self.label

    class Admin:
	pass



class DNA( models.Model ):
    """
    DNA holds the information about a physical piece of DNA like,
    in most cases, a plasmid + biobrick.

    Thus it needs to have a sequence, and, earlier or later,
    it should be stored in one or more places.

    The samples holding this DNA are accessible via the field
    'samples' which is imported from the DNA table.
    """
    DNA_TYPES = (('plasmid','Plasmid'), ('oligo','Oligo') )

    name = models.CharField( maxlength=15, primary_key=True)

    #: [optional] the parts.mit.edu ID
    mit_id = models.CharField( maxlength=20, blank=True )

    #: [optional]
    description = models.CharField( maxlength=200, blank=True )
    
    dna_type = models.CharField( 'type of DNA', maxlength=30,
				 choices=DNA_TYPES )

    #: link to biobrick definition
    #: [optional] to allow planning
    biobrick = models.ForeignKey( 'Biobrick',
				  verbose_name='with biobrick',
				  related_name='implementations',
				  blank=True,
                                  null=True)

    #: link to host vector
    #: [optional] to allow planning and oligos or cells
    vector = models.ForeignKey( 'Vector',
                                verbose_name='in vector',
                                related_name='implementations',
                                blank=True,
                                null=True)

    #: Strain this DNA is stored in, if any
    cell = models.ForeignKey( 'Biobrick',
                              verbose_name='in cell',
                              related_name='hosted_biobricks',
                              blank=True,
                              null=True)

    sequence = models.TextField()

    #: [optional]
    genebank_sequence = models.TextField('Genebank-formatted sequence',
					 blank=True)

    #: [optional]
    comments = models.TextField( blank=True )

    # [optional] links_html
    references = models.ManyToManyField( Reference, blank=True )

    def __unicode__( self ):
	return self.name


    class Admin:

	fields = (
	    (None, {
	    'fields': ('name','mit_id', 'description', 'dna_type')
	    }),
	    ('Details',{
		'fields' : ('biobrick', 'vector', 'cell', 'sequence',)
	    }),
	    ('Additional information',{
	    'fields': ('comments','references', 'genebank_sequence'),
            'classes':'collapse'
	    }),
	)

    class Meta:

	verbose_name = 'physical DNA'


class BrickType( models.Model ):
    """
    Describes the different types of Biobricks that are defined by
    the MIT repository. The type of a biobrick should enter the name
    with a type code letter:
    C .. protein coding; G .. primer; M .. tag; P .. protein generator;
    S .. intermediate; V .. cell strain; B .. basic or general part;
    See: http://parts.mit.edu/registry/index.php/Help:BioBrick_Part_Names
    """
    name = models.CharField( maxlength=50, primary_key=True )
    
    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )

    #: obligate type code letter that enters the name of the part
    type_code = models.CharField('1-letter type code', maxlength=1)

    def __unicode__(self):
	return self.type_code + ':' + self.name

    class Admin:
	pass


class BrickCategory( models.Model ):
    """
    Describes the different functional types of biobricks.
    (regulators, protein, transcription, etc.)
    """
    name = models.CharField( maxlength=50, primary_key=True )
    
    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )


    def __unicode__(self):
	return self.name

    class Admin:
	pass

    class Meta:
	verbose_name_plural = 'BrickCategories'


class ProgressCategory( models.Model ):
    """
    A customized set of progress tags like perhaps 'synthesis pending',
    'sequenced'?
    """

    name = models.CharField( maxlength=50, primary_key=True )
    
    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )


    def __unicode__(self):
	return self.name

    class Admin:
	pass

    class Meta:
	verbose_name_plural = 'ProgressCategories'
    

class Biobrick( models.Model ):
    """
    A Biobrick holds the description of a biobrick sequence.

    Note: The connection from a Biobrick to its physical DNA implementations
    is established by a relation 'biobrick' in DNA.
    """
    BRICK_FORMAT_CHOICES = ( ('classic', 'classic biobrick format'),
			   ('biofusion', 'protein biobrick / Biofusion'),
			   ('non_standard', 'non-standard (comment!)'),
			   ('unknown', 'unknown format (comment!)') )

    STATUS_CHOICES = ( ('available', 'available'),
		       ('planning', 'planning'),
		       ('submitted', 'submitted to MIT') )

    EXPERIENCE_CHOICES = ( ('works', 'works'),
			   ("worksnot", "doesn't work"),
			   ('none', 'no experience') )


    name = models.CharField( maxlength=15, primary_key=True )

    #: [optional] ID fetched from parts.mit.edu
    mit_id = models.CharField( maxlength=20, blank=True )

    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )

    #: obligate
    brick_format = models.CharField( 'format of Biobrick', maxlength=30,
				   choices=BRICK_Format_CHOICES,
				   default='classic' )

    #: obligate
    brick_type = models.ForeignKey( 'type of Biobrick',
                                    verbose_name='biobrick type',
                                    related_name='child_biobricks')

    #: link to table of functional categories
    categories = models.ManyToManyField( BrickCategory )

    status = models.CharField( 'implementation status', maxlength=30,
			       choices=STATUS_CHOICES,
			       default='planning')


    experience = models.CharField( maxlength=30,
				   choices=EXPERIENCE_CHOICES,
				   default='none' )

    #: [optional]
    progress = models.ManyToManyField( ProgressCategory,
				       blank=True )
	
    #: [optional]
    comments = models.TextField( blank=True )

    sequence = models.TextField()

    #: [optional]
    genebank_sequence  = models.TextField('Genebank-formatted sequence',
					 blank=True)

    #: [optional]
    source_gid = models.CharField( 'source genebank entry', maxlength=10,
				   blank=True )

    #: [optional]
    source_txt = models.CharField( 'source free text', maxlength=200,
				   blank=True )

    # [optional] links_html
    references = models.ManyToManyField( Reference, blank=True )

    def __unicode__(self):
	return self.name


    class Admin:

	fields = (
	    (None,
             {'fields': ('name','mit_id', 'short_description',
                         'status', 'experience')}),
	    ('Details',
             {'fields' : ('brick_format', 'brick_type', 'categories',
                          'description')}),
	    ('Additional information',
	     {'fields': ('progress', 'source_gid', 'source_txt',
			 'references', 'comments',
			 'genebank_sequence'),
	       'classes':'collapse'}),
	    ('Sequence',
             {'fields':('sequence',)}),
	    )


class SelectiveMarker( models.Model ):
    """
    Describes an Antibiotic or similar resistence marker
    """
    name = models.CharField( maxlength=15, primary_key=True )

    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )

    # [optional] links_html
    references = models.ManyToManyField( Reference, blank=True )
    

    def __unicode__(self):
	return self.name

    class Admin:
        pass


class Vector( models.Model ):
    """
    A vector describes a Plasmid (are there non-plasmid vectors?)
    """
    name = models.CharField( maxlength=15, primary_key=True )

    #: [optional] ID fetched from parts.mit.edu
    mit_id = models.CharField( maxlength=20, blank=True )

    #: obligate brief description
    short_description = models.CharField( maxlength=200 )

    #: [optional] long description
    description = models.TextField( 'detailed description', blank=True )

    sequence = models.TextField()

    #: [optional] link to resistence or similar marker
    marker = models.ManyToManyField( SelectiveMarker,
                                     verbose_name='selective marker(s)',
                                     null=True,
                                     blank=True)

    #: [optional]
    comments = models.TextField( blank=True )

    # [optional] links_html
    references = models.ManyToManyField( Reference, blank=True )

    #: [optional]
    genebank_sequence = models.TextField('Genebank-formatted sequence',
                                         blank=True)
    def __unicode__(self):
	return self.name

    class Admin:
        
	fields = (
            (None,
             { 'fields': ('name','mit_id', 'short_description',)}),
            ('Details',
             {'fields': ('description', 'marker', 'sequence')}),
            ('Additional information',
             {'fields': ('comments', 'references', 'genebank_sequence'),
              'classes':'collapse' }),
            )
            
######################################
## register models to databrowsing app

databrowse.site.register( Biobrick )
databrowse.site.register( Vector )
databrowse.site.register( DNA )
databrowse.site.register( StorageContainer )
databrowse.site.register( Sample )
databrowse.site.register( BrickCategory )
databrowse.site.register( SelectiveMarker )
databrowse.site.register( Reference )
