#from django.db.backends.db2_9.base import quote_name
from django.conf import settings

def get_table_list(cursor):
	""" Here's the tricky part. The tables are accessed by their schema
	for example, all the systems tables are prefixed by SYSIBM as their schema.
	Since we can get really *lots* of tables without filtering by creator this can get
	really messy, So I've used DATABASE_SCHEMA to filter it.
	RTRIMs are needed because the fields return in full column size filled with spaces.
	Using the SYSIBM.TABLES VIEW.
	"""
	from django.conf import settings
	from django.utils.encoding import smart_str, force_unicode
	try:
		schema = settings.DATABASE_SCHEMA.upper()
	except:
		schema = ''
	if schema != '':
		cursor.execute("""SELECT RTRIM(table_name) as TABLES, RTRIM(table_schema) as SCHEMA FROM SYSIBM.TABLES 
		WHERE TABLE_SCHEMA = %s AND TABLE_TYPE = 'BASE TABLE' ORDER BY table_name""", [schema])
	else: # Fallback to everything but SYS*
		cursor.execute("""SELECT RTRIM(table_name) as TABLES, RTRIM(table_schema) as SCHEMA FROM SYSIBM.tables 
		WHERE table_schema NOT LIKE 'SYS%%' AND TABLE_TYPE = 'BASE TABLE' ORDER BY table_name""", [])
	rows = cursor.fetchall()
	res = []
	# for tables from the user: do not append schema, others: use SCHEMA.TABLE as name
	for row in rows:
		if row[1].upper() == settings.DATABASE_USER.upper():
			res.append(smart_str(row[0].upper()))
		else:
			res.append(smart_str("%s.%s" % (row[1].upper(), row[0].upper())))
	return res

def get_table_description(cursor, table_name):
	"Returns a description of the table with the DB-API cursor.description interface."
	cursor.execute("SELECT * FROM %s FETCH FIRST 1 ROWS ONLY" % (table_name,))
	return cursor.description

def _name_to_index(cursor, table_name):
    """
    Returns a dictionary of {field_name: field_index} for the given table.
    Indexes are 0-based.
    """
    return dict([(d[0], i) for i, d in enumerate(get_table_description(cursor, table_name))])

def get_relations(cursor, table_name):
	"""
	Returns a dictionary of {field_index: (field_index_other_table, other_table)}
	representing all relationships to the given table. Indexes are 0-based.
	Beware, PyDB2 returns the full length of the field, filled with spaces, 
	I have to .strip() every single string field >.<
	"""
	from django.conf import settings
	
	if table_name.count('.') == 1:
		schema, tn = table_name.split('.')
	else:
		tn = table_name
		schema = settings.DATABASE_USER
	cursor.execute("""SELECT fk.ORDINAL_POSITION, pk.ORDINAL_POSITION, pk.TABLE_NAME
			FROM SYSIBM.SQLFOREIGNKEYS r, SYSIBM.SQLCOLUMNS fk, SYSIBM.SQLCOLUMNS pk
			WHERE r.FKTABLE_SCHEM = %s AND r.FKTABLE_NAME = %s
			AND r.FKTABLE_SCHEM = fk.TABLE_SCHEM AND r.FKTABLE_NAME = fk.TABLE_NAME
			AND r.FKCOLUMN_NAME = fk.COLUMN_NAME
			AND r.PKTABLE_SCHEM = pk.TABLE_SCHEM AND r.PKTABLE_NAME = pk.TABLE_NAME
			AND r.PKCOLUMN_NAME = pk.COLUMN_NAME
		""", (schema.upper(), tn))
		
	relations = {}
	for row in cursor.fetchall():
		relations[int(row[0]) - 1] = (int(row[1]) -1, row[2].strip())
	return relations
	
def get_indexes(cursor, table_name):
	"""
	Returns a dictionary of fieldname -> infodict for the given table,
	where each infodict is in the format:
		{'primary_key': boolean representing whether it's the primary key,
		'unique': boolean representing whether it's a unique index}
	DB2 is weird here, like this seems to be ok but I don't get 100% of the indexes
		syscolumns.keyseq means the column part of a "parent key".
		sysindexes.uniquerule == P means it's primary, U == unique,
			and D == duplicates allowed.
	"""	
	from django.conf import settings
	if table_name.count('.') == 1:
		schema, tn = table_name.split('.')
	else:
		tn = table_name
		schema = settings.DATABASE_USER
	cursor.execute("""SELECT c.COLUMN_NAME, i.UNIQUERULE 
			FROM SYSIBM.SQLCOLUMNS c, SYSCAT.INDEXES i, SYSCAT.INDEXCOLUSE k
			WHERE c.TABLE_SCHEM = %s AND c.TABLE_NAME = %s
			AND C.TABLE_SCHEM = i.TABSCHEMA AND c.TABLE_NAME = i.TABNAME
			AND i.INDSCHEMA = k.INDSCHEMA AND i.INDNAME = k.INDNAME
			AND c.COLUMN_NAME = k.COLNAME
		""", (schema.upper(), tn))

	indexes = {}
	for row in cursor.fetchall():
		urule = row[1].strip()
		name = row[0].strip()
		if urule == "P":
			# Override everything, as this is a primary key.
			indexes[name] = {'primary_key':True, 'unique':False}
		elif urule == "U":
			try:
				if indexes[name]['primary_key'] == True:
					# Can appear twice, but primary is primary anyway.
					continue
				else:
					indexes[name] = {'primary_key':False, 'unique':True}
			except: # TODO: Only a keyerror can happen here, right?
				indexes[name] = {'primary_key':False, 'unique':True}
		else: # urule = "D" not sure if there are others.
			try:
				# Should never happen, but...
				if indexes[name]['primary_key'] == True or indexes[name]['unique'] == True:
					continue
				else:
					indexes[name] = {'primary_key':False, 'unique':False}
			except: # TODO: same as above ^_^
				indexes[name] = {'primary_key':False, 'unique':False}
	return indexes

DATA_TYPES_REVERSE = {
	-99:'TextField', # CLOB
	-98:'TextField', # BLOB
	-97:'TextField', # Long VarGraphic
	-96:'TextField', # VarGraphic
	-95:'TextField', # Graphic
	-5: 'IntegerField', # Big Int
	-4: 'TextField', # Binary Long VarChar
	-3: 'TextField', # Binary VarChar
	-2: 'TextField', # Binary
	-1: 'TextField', # Long VarChar
	1: 'CharField',  # Char
	2: 'FloatField', # Numeric
	3: 'FloatField', # Decimal
	4: 'IntegerField', # Integer
	5: 'BooleanField', # SmallInt
	6: 'FloatField', # Float
	7: 'FloatField', # Real
	8: 'FloatField', # Double
	12: 'CharField', # VarChar
	91: 'DateField', # Date
	92: 'TimeField', # Time
	93: 'DateTimeField', # TimeStamp
}
