"""
Syndication feed generation library -- for RSS 2.0 and Atom 1.0

Requires:

* ElementTree <http://effbot.org/zone/element-index.htm>

Changes from original django feedgenerator:
* Added author_name, author_email, author_link, categories and subtitle 
  parameters to SyndicationFeed
* Renamed writeString to write_string.
* Uses ElementTree rather than SimpleXMLWriter.
* DefaultRssFeed is renamed to DefaultFeed.
* DefaultFeed is set to Atom10Feed.

Notes:

* All string input values need to be unicode objects.
* Date objects need to be Python datetime objects.
* Only RSS 2.01 and Atom 1.0 is implemented.
* DefaultFeed is Atom10Feed

* For RSS/Atom feeds, feed publication date will be set as the most recent
  item date.
* For Atom feeds, we only populate the <summary> fields for each item.
* For Atom feeds, if not unique_id is supplied, a Tag URI is generated:
  <http://diveintomark.org/archives/2004/05/28/howto-atom-id>
* For Atom feeds, description field will be parsed as XHTML and embedded inline.
  If it fails to validate as XML/XHTML, it will be escaped and embedded as HTML.

Example:

>>> feed = feedgenerator.Atom10Feed(
...     title=u"Another Boring Blog",
...     link=u"http://anotherboringblog.com/",
...     description=u"A group weblog by the sharpest minds in online media/journalism/publishing.",
...     feed_url = u"http://anotherboringblog.com/atom.xml",
...     author_name = u"A. Blogger",
...     author_email = u"a.blogger@anotherboringblog.com",
...     categories = [u"blog", u"boring"]
... )
>>> feed.add_item(
...     title=u"What I had for breakfast", 
...     link=u"http://anotherboringblog.com/what-i-had-for-breakfast/",
...     description=u"Eggs, Bacon and Potatoes.",
...     pubdate = datetime.now(),
...     categories = [u"food"]
..      )
>>> print feed.writeString('utf-8')

"""

from elementtree.ElementTree import Element, SubElement, ElementTree, fromstring, tostring
from xml.parsers.expat import ExpatError
import re
import email
import time
from datetime import datetime

ATOM_NS = 'http://www.w3.org/2005/Atom'
XHTML_NS = 'http://www.w3.org/1999/xhtml'

class Enclosure:
    "Represents an RSS enclosure"
    def __init__(self, url, length, mime_type):
        "All args are expected to be Python Unicode objects"
        self.url, self.length, self.mime_type = url, length, mime_type

class SyndicationFeed:
    "Base class for syndication feeds. Subclasses should provide write_string()"
    def __init__(self, title, link, description, feed_url,
                author_name = None, author_email = None, author_link = None,
                categories = [], subtitle = None, language = None):
        self.feed = {'title': title, 
                    'link': link, 
                    'feed_url': feed_url,
                    'description': description,
                    'author_name': author_name, 
                    'author_email': author_email,
                    'author_link': author_link,
                    'categories': categories,
                    'subtitle': subtitle,
                    'language': language
        }
        self.items = []
                     
    def add_item(self, title, link, description, 
                author_name = None, author_email = None, pubdate = None,
                comments = None, unique_id = None, enclosure = None,
                categories = []):
        """
        Adds an item to the feed. All args are expected to be Python Unicode
        objects except:
        
        pubdate     - DateTime object.
        enclosure   - Enclosure object.
        categories  - List of Unicode objects.
        """        
        self.items.append({
            'title': title,
            'link': link,
            'description': description,
            'author_email': author_email,
            'author_name': author_name,
            'pubdate': pubdate,
            'comments': comments,
            'unique_id': unique_id,
            'enclosure': enclosure,
            'categories': categories,
        })
        
    def count(self):
        return len(self.items)
        
    def write(self, f, encoding):
        """ Writes output to a file object with specified encoding """
        f.write(self.write_string(encoding))
        
    def latest_post_date(self):
        """ Get the latest item date otherwise just return the current date """
        updates = [i['pubdate'] for i in self.items if (i['pubdate'] is not None)]
        if len(updates) > 0:
            updates.sort()
            return updates[-1]
        else:
            return datetime.now()
        

class Rss201Feed(SyndicationFeed):
    " RSS 2.01 rev 2 feed <http://blogs.law.harvard.edu/tech/rss>"

    def rfc2822_date(self, date):
        return email.Utils.formatdate(time.mktime(date.timetuple()))
        
    def write_string(self, encoding):
        tree = Element('rss', {'version':'2.0'})
        channel = self.write_channel_header(tree)
        self.write_items(channel)
        return tostring(tree, encoding = encoding)
        
    def write_channel_header(self, tree):
        channel = SubElement(tree, 'channel')
        SubElement(channel, 'title').text = self.feed['title']
        SubElement(channel, 'link').text =  self.feed['link']
        SubElement(channel, 'description').text =  self.feed['description']
        
        if self.feed['language'] is not None:
            SubElement(channel, 'language').text =  self.feed['language']
            
        if self.feed['categories'] is not []:
            for cat in self.feed['categories']:
                SubElement(channel, 'category').text = cat
                
        SubElement(channel, 'pubDate').text = self.rfc2822_date(self.latest_post_date())
        
        return channel
        
    def write_items(self, channel):
        for item in self.items:
            entry = SubElement(channel, 'item')
            SubElement(entry, 'title').text = item['title']
            SubElement(entry, 'link').text = item['link']
            if item['description'] is not None:
                SubElement(entry, 'description').text = item['description']
            if item['author_name'] is not None and item['author_email'] is not None:
                SubElement(entry, 'author').text = u'%s (%s)' % (item['author_email'], item['author_name'])
            if item['pubdate'] is not None:
                SubElement(entry, 'pubDate').text = self.rfc2822_date(item['pubdate'])
            if item['unique_id'] is not None:
                SubElement(entry, 'guid').text = item['unique_id']
            else:
                SubElement(entry, 'guid').text = item['link']
            if item['enclosure'] is not None:
                SubElement(entry, 'enclosure', {'url': item['enclosure'].url,
                                                'length': item['enclosure'].length,
                                                'type': item['enclosure'].mime_type})
            for cat in item['categories']:
                SubElement(entry, 'category').text = cat
        
        
class Atom10Feed(SyndicationFeed):
    """ An Atom 1.0 Feed 
        <http://atompub.org/2005/07/11/draft-ietf-atompub-format-10.html> 
    """
    
    def rfc3339_date(self, date):
        return date.strftime('%Y-%m-%dT%H:%M:%SZ')
        
    def tag_uri(self, url, date):
        """ Creates a TagURI 
            <http://diveintomark.org/archives/2004/05/28/howto-atom-id>
        """
        tag = re.sub('^http://', '', url)
        tag = re.sub('/', ',%s:/' % date.strftime('%Y-%m-%d'), tag, 1)
        tag = re.sub('#', '/', tag)
        return 'tag:' + tag        
    
    def write_string(self, encoding):
        tree = Element('feed', {'xmlns':ATOM_NS})
        self.write_feed_header(tree)
        self.write_items(tree)
        return tostring(tree, encoding = encoding)
        
    def write_feed_header(self, tree):
        SubElement(tree, 'title').text = self.feed['title']
        SubElement(tree, 'link', {'rel':'alternate', 'href': self.feed['link']})
        SubElement(tree, 'link', {'rel':'self', 'href':self.feed['feed_url']})
        SubElement(tree, 'id').text = self.feed['link']        
        
        if self.feed['author_name'] is not None:
            author = SubElement(tree, 'author')
            SubElement(author, 'name').text = self.feed['author_name']
            if self.feed['author_email'] is not None:        
                SubElement(author, 'email').text = self.feed['author_email']
            if self.feed['author_link'] is not None:
                SubElement(author, 'email').text = self.feed['author_link'] 
                   
        if self.feed['subtitle'] is not None:
            SubElement(tree, 'subtitle').text = self.feed['subtitle']
        

        SubElement(tree, 'updated').text = self.rfc3339_date(self.latest_post_date())
            
        for cat in self.feed['categories']:
            SubElement(tree, 'category', {'term':cat})

    def write_items(self, tree):
        for item in self.items:
            entry = SubElement(tree, 'entry')
            SubElement(entry, 'title').text = item['title']
            SubElement(entry, 'link', {'rel':'alternate', 'href': item['link']})
            SubElement(entry, 'updated').text = self.rfc3339_date(item['pubdate'])
                        
            if item['unique_id'] is not None:
                SubElement(entry, 'id').text =  item['unique_id']
            else:
                tag_uri = self.tag_uri(item['link'], item['pubdate'])
                SubElement(entry, 'id').text =  tag_uri
                
            # here we do something cunning, try parsing fragment as xml
            # and if it works, we just put it in, otherwise, just
            # use type="html" and escape the contents
            try:
                fragment = fromstring('<div>%s</div>' % item['description'].encode('utf-8'))
                fragment.attrib['xmlns'] = XHTML_NS
                SubElement(entry, 'summary', {'type':'xhtml'}).append(fragment)
            except ExpatError:
                SubElement(entry, 'summary', {'type':'html'}).text = item['description']

            if item['enclosure'] is not None:
                SubElement(entry, 'link', 
                    {'rel':'enclosure', 
                    'href':item['enclosure'].url, 
                    'length':item['enclosure'].length, 
                    'type':item['enclosure'].mime_type})
                
            for cat in item['categories']:
                SubElement(entry, 'category', {'term':cat})
                
            if item['author_name'] is not None:
                author = SubElement(entry, 'author')
                SubElement(author, 'name').text = item['author_name']
                if item['author_email'] is not None:        
                    SubElement(author, 'email').text = item['author_email']

DefaultFeed = Atom10Feed

                