import sys, os
from django.conf.urls import defaults

"""
Automatic site configuration.

In your settings module:: 
    
    import autosite
    ROOT_URLCONF = autosite.root_urlconf('sitename')
    TEMPLATE_DIRS = autosite.template_dirs('sitename')
    INSTALLED_APPS = autosite.installed_apps('sitename')

In your site-level url configuration module, which thanks to `root_urlconf` 
can be in ``sitename/urls.py`` 
rather than ``sitename/settings/urls/main.py``, you can use `sitepatterns`
rather than `patterns` to do most of the work for you:: 

    from autosite import sitepatterns
    urlpatterns = patterns('', 'sitename',
        (r'^/?$', 'sitename.views.root'),
        # ... any more explicit site-level patterns and destinations... 
        )

`sitepatterns` will iterate through the application packages looking 
for app-level url configuration modules (either ``appname/urls.py`` or 
``appname/urls/app.py``) and automatically add them to the site's pattern 
list with the pattern ``r'^appname/'``. 

If a app-level url configuration module can't be found, `sitepatterns` 
will iterate through ``sitename.apps.appname.views`` (whether it's a module 
or a package) looking for callables with a ``.urlpattern`` attribute 
(for which the value should be a regular expression) or a ``.urlpatterns`` 
attribute (for which the value should be a list of regular expressions). 

For Python 2.3, you should set ``.urlpattern`` manually. For Python 2.4, 
the ``urlpattern`` decorator will do it for you::

    @urlpattern(r'^/?$')
    def index(request): 
       # ...

Note that the view modules' expressions will be added in the alphabetic 
order of their function names. If you were depending on careful ordering 
of your pattern list, either keep using your manual url configuration 
module or use the ``(?<!...)`` negative lookbehind assertion. 

The practical upshot of using autosite is that you type less and don't 
have to maintain quite so many files. Rather than::

    sitename/settings/__init__.py
    sitename/settings/main.py
    sitename/settings/admin.py
    sitename/settings/urls/main.py
    sitename/apps/appname/urls/__init__.py
    sitename/apps/appname/urls/appname.py
    sitename/apps/appname/views/__init__.py
    sitename/apps/appname/views/appname.py
    
... you can consolidate back to: 

    sitename/settings.py
    sitename/urls.py
    sitename/apps/appname/views.py

This module has been brought to you by one programmer's bizarre tendency to 
to spend two hours writing 300+ lines of code to replace around 30 lines of 
code that were taking him less than two minutes per day to maintain. His 
insanity is your gain. If only 100 Django programmers benefit from this 
module, all his hard work will have been worthwhile. 
"""

__all__ = [
    'urlconf', 
    'patterns', 
    'template_dirs',
    'installed_apps',
    'sitepatterns',
    'apppatterns',
    'urlpattern',
    ]

def splitall(path): 
    """Split `path` into all of its components."""
    segments = []
    base = path
    while True:    
        base, name = os.path.split(base)
        if name: 
            segments.append(name)
        else: 
            if base:
                segments.append(base)
            segments.reverse()
            return segments

def child_packages_and_modules(
        packagename, 
        relative=True, 
        onlypackages=False, 
        returndirs=False, 
        depth=0): 
    """Return a list of packages and modules under `packagename`. 

    relative -- exclude the package name itself from the results
    onlypackages -- exclude modules (*.py) from the results
    returndirs -- return the path, not the module/package name
    depth -- if not 0, limit the result depth
    """
    
    # Find the package
    modstack = packagename.split('.')
    package = __import__(packagename, {}, {}, modstack[-1])
    packagedir, initfile = os.path.split(package.__file__)
    packagedir = os.path.abspath(packagedir)
    assert initfile in [ '__init__.py', '__init__.pyc' ]
    assert os.path.isdir(packagedir)
    packagedirstack = splitall(packagedir)
    lpds = len(packagedirstack)
    results = []
    
    # Define a helpful 'append' method so we don't have to duplicate 
    # this logic.
    def append(segments): 
        if depth and len(segments) > depth: 
            return
        if returndirs: 
            joiner = lambda l: os.path.join(*l)
            basesegments = packagedirstack
        else: 
            joiner = '.'.join
            basesegments = modstack
        if relative and not returndirs: 
            if segments: 
                results.append(joiner(segments))
        else: 
            results.append(joiner(basesegments + segments))

    # Walk the package directory, gathering results.
    for dirpath, dirnames, filenames in os.walk(packagedir):
        dirstack = splitall(dirpath)
        assert dirstack[:lpds] == packagedirstack
        relsegments = dirstack[lpds:]
        heremodstack = modstack + relsegments
        heredirstack = packagedirstack + relsegments
        if not ('__init__.py' in filenames or '__init__.pyc' in filenames): 
            # wherever we are, it isn't a package and we shouldn't 
            # recurse any deeper
            del dirnames[:]
        else: 
            append(relsegments)
            if not onlypackages:
                for filename in filenames:
                    name, ext = os.path.splitext(filename)
                    if ext == '.py' and name != '__init__': 
                        append(relsegments + [name])
    return results

def urlconf(modulename, lookfor=None, verify=False): 
    """Find the urlconf for `modulename`, which should be the name of the 
    site or one of its applications (``"sitename.appname"``).
    
    lookfor -- list of submodules to look for
    """

    if lookfor is None: 
        segments = modulename.split('.')
        if 'apps' in segments: 
            # sitename.apps.appname
            lookfor = ['urls.%s' % segments[-1], 'urls']
        else: 
            # sitename
            lookfor = ['settings.urls.main', 'urls']
    submodules = child_packages_and_modules(modulename)
    fails = []
    for submodule in lookfor: 
        modname = '%s.%s' % (modulename, submodule)
        if submodule in submodules: 
            if verify: 
                module = __import__(modname, {}, {}, 'urlconf_module')
                if hasattr(module, 'urlpatterns'): 
                    return modname
                else: 
                    raise AssertionError, "We thought %s was a urlconf " \
                            "module, but couldn't find urlpatterns in it." % (
                                    modname)
            else: 
                return modname
        else: 
            fails.append(modname)
    raise AssertionError, "Couldn't find a url module amongst %s." % (
            ', '.join(fails))
root_urlconf = urlconf
         
def template_dirs(sitemodulename, lookfor=['templates'], incadmin=True): 
    """Find all the template/ directories in `sitemodulename`.
    
    lookfor -- adjust what subdirectory names to look for
    incadmin -- if True (default), include the administration templates."""

    results = []
    for packagedir in child_packages_and_modules(sitemodulename, 
            onlypackages=True, returndirs=True): 
        for subdir in lookfor: 
            templatedir = os.path.join(packagedir, subdir)
            if os.path.isdir(templatedir): 
                results.append(templatedir)
    if incadmin:
        return results + template_dirs('django.conf', 
                lookfor=['admin_templates'],
                incadmin=False)
    return results

def installed_apps(sitemodulename): 
    """Find all the apps in `sitemodulename`."""
    return child_packages_and_modules('%s.apps' % sitemodulename, 
            onlypackages=True, depth=1, relative=False)[1:]

def sitepatterns(basename, sitename, incadmin=True, *patlist): 
    """Like django.conf.urls.defaults.patterns, but automatically includes 
    the urlconfs for any applications it can find.
    
    basename -- hopefully, used only for patlist
    sitename -- the name of the site module to scan for apps and urlconfs
    incadmin -- include the administration site.
    """

    patlist = list(patlist)
    for appmodname in installed_apps(sitename): 
        appname = appmodname.split('.')[-1]
        try: 
            uc = urlconf(appmodname)
            pattern = (
                    r'^%s/' % appname, 
                    defaults.include(urlconf(appmodname))
                    )
            patlist.append(pattern)
        except AssertionError: # Couldn't find it! 
            for urlpattern, target in _apppatterns(appmodname): 
                if urlpattern[:1] == '^': 
                    urlpattern = urlpattern[1:]
                pattern = (
                        r'^%s/%s' % (appname, urlpattern),
                        target
                        )
                patlist.append(pattern)
    if incadmin: 
        pattern = (r'^admin/', defaults.include('django.conf.urls.admin'))
        patlist.append(pattern)
    import pprint
    pprint.pprint(patlist)
    return defaults.patterns(basename, *patlist)
    
def _apppatterns(appmodname): 
    """Does the heavy lifting for `apppatterns`."""
    patlist = []
    for modname in child_packages_and_modules(appmodname): 
        if modname == 'views' or modname.startswith('views.'): 
            fullmodname = '%s.%s' % (appmodname, modname)
            module = __import__(fullmodname, {}, {}, fullmodname)
            for attname in dir(module): 
                att = getattr(module, attname)
                if callable(att): 
                    if hasattr(att, 'urlpattern'): 
                        pattern = (
                                getattr(att, 'urlpattern'), 
                                '%s.%s' % (fullmodname, attname)
                                )
                        patlist.append(pattern)
                    elif hasattr(att, 'urlpatterns'): 
                        for urlpattern in getattr(att, 'urlpatterns'): 
                            pattern = (
                                    urlpattern, 
                                    '%s.%s' % (fullmodname, attname)
                                    )
                            patlist.append(pattern)
    return patlist
    
def apppatterns(basename, appmodname, *patlist): 
    """Like django.conf.urls.defaults.patterns, but looks for views modules 
    and scans them for view modules with `urlpattern` attributes.
    
    basename -- only used for patlist
    appmodname -- the name of the app in ``sitename.apps.appname`` format.
    """

    if basename: 
        # Incorporate basename so we don't have to pass it down.
        patlist = [(urlpattern, '%s.%s' % (basename, target))
                   for urlpattern, target in patlist[:]]
    else:
        # Just make a list of it.
        patlist = list(patlist) 
    patlist.extend(_apppatterns(appmodname))
    return defaults.patterns('', *patlist)

def urlpattern(urlp): 
    """Decorate a view function with a URL pattern to be picked up by 
    the automatic stuff above."""
    def decorator(func): 
        if hasattr(func, 'urlpatterns'): 
            func.urlpatterns.append(urlp)
        else: 
            func.urlpatterns = [urlp]
        return func
    return decorator
