"""
Spain-specific Form helpers
"""
from django.newforms import ValidationError
from django.newforms.fields import Field, EMPTY_VALUES
from django.utils.translation import ugettext_lazy as _

class ESNifField(Field):
	"""
	Spanish NIF/NIE/CIF (Fiscal Identification Number) code.

	Validates three diferent formats,
		NIF (individuals): 12345678A
		CIF (companies): A12345678
		NIE (foreigners): X12345678A
	according to the simple checksum algorithm.
	Value can include any separator between number and letter (commonly used hyphen or space). Actually any strange character is ignored in whole value.
	Less (or more) than 8 digits are allowed. It's not a mistake, old NIFs have 7 numbers, and 9 number codes could exist in the future.
	Return value is normalized like patterns in previous examples (also changing letters to upper case).
	"""
	def clean(self, value):
		super(ESNifField, self).clean(value)
		if value in EMPTY_VALUES:
			return ''
		checksum_list = 'TRWAGMYFPDXBNJZSQVHLCKE'
		normalized_value = ''
		for char in value:
			if char.isdigit() or char.upper() in checksum_list:
				normalized_value += char.upper()
		if normalized_value[0] == 'X' and normalized_value[1:-1].isdigit() and normalized_value[-1] in checksum_list:
			nif_number, nif_letter = normalized_value[1:-1], normalized_value[-1]
		elif normalized_value[0] in checksum_list and normalized_value[1:].isdigit():
			nif_letter, nif_number = normalized_value[0], normalized_value[1:]
		elif normalized_value[:-1].isdigit() and normalized_value[-1] in checksum_list:
			nif_number, nif_letter = normalized_value[:-1], normalized_value[-1]
		else:
			print normalized_value
			raise ValidationError(_('Enter a CIF, NIF or NIE in the format Z99999999, 99999999Z or X99999999Z.'))

		if nif_letter == checksum_list[int(nif_number) % len(checksum_list)]:
			return normalized_value
		else:
			raise ValidationError(_('Invalid letter for CIF/NIF number'))

