from django.db import models

class Category(models.Model):
    name = models.CharField(maxlength=20)
    url = models.CharField('The URL', maxlength=40)

    def __str__(self):
        return self.name

class Writer(models.Model):
    name = models.CharField(maxlength=50, help_text='Use both first and last names.')

    def __str__(self):
        return self.name

class Article(models.Model):
    headline = models.CharField(maxlength=50)
    pub_date = models.DateField()
    created = models.DateField(editable=False)
    writer = models.ForeignKey(Writer)
    article = models.TextField()
    categories = models.ManyToManyField(Category, blank=True)

    def save(self):
        import datetime
        if not self.id:
            self.created = datetime.date.today()
        return super(Article, self).save()

    def __str__(self):
        return self.headline

class PhoneNumber(models.Model):
    phone = models.PhoneNumberField()
    description = models.CharField(maxlength=20)

    def __str__(self):
        return self.phone

def CategoryCallback(f, **kwargs):
    if not "cat_" in f.name:
      f.name = 'cat_'+f.name
    return f.formfield(**kwargs)
  

__test__ = {'API_TESTS': """
>>> from django.newforms import form_for_model, form_for_instance, save_instance, BaseForm, Form, CharField
>>> import datetime

>>> Category.objects.all()
[]

>>> CategoryForm = form_for_model(Category, formfield_callback=CategoryCallback)
>>> f = CategoryForm()
>>> print f
<tr><th><label for="id_cat_name">Name:</label></th><td><input id="id_cat_name" type="text" name="cat_name" maxlength="20" /></td></tr>
<tr><th><label for="id_cat_url">The URL:</label></th><td><input id="id_cat_url" type="text" name="cat_url" maxlength="40" /></td></tr>
>>> print f.as_ul()
<li><label for="id_cat_name">Name:</label> <input id="id_cat_name" type="text" name="cat_name" maxlength="20" /></li>
<li><label for="id_cat_url">The URL:</label> <input id="id_cat_url" type="text" name="cat_url" maxlength="40" /></li>
>>> print f['name']
Traceback (most recent call last):
...
KeyError: "Key 'name' not found in Form"
>>> print f['cat_name']
<input id="id_cat_name" type="text" name="cat_name" maxlength="20" />

# this works, if there is no ForeignKeys
>>> Category(name="foo", url="http://www.example.com")
<Category: foo>

# this works
>>> w = Writer(name='Bob Woodward')
>>> Article(headline='Test article', pub_date=datetime.date(1988, 1, 4), writer=w, article='Hello.')
<Article: Test article>

# now lets try one with ForeignKey
>>> ArticleForm = form_for_model(Article, formfield_callback=CategoryCallback)
>>> f = ArticleForm(auto_id=False)
>>> print f
<tr><th>Headline:</th><td><input type="text" name="cat_headline" maxlength="50" /></td></tr>
<tr><th>Pub date:</th><td><input type="text" name="cat_pub_date" /></td></tr>
<tr><th>Writer:</th><td><select name="cat_writer">
<option value="" selected="selected">---------</option>
</select></td></tr>
<tr><th>Article:</th><td><textarea name="cat_article"></textarea></td></tr>
<tr><th>Categories:</th><td><select multiple="multiple" name="cat_categories">
</select><br /> Hold down "Control", or "Command" on a Mac, to select more than one.</td></tr>
>>> w = Writer(name='Bob Woodward')

#but now the same one fails, even though Article object as itself should not be modified by for_for_model
>>> Article(headline='Test article', pub_date=datetime.date(1988, 1, 4), writer=w, article='Hello.')
"""}