# Django psycopg connection pool + fork() bug reproduction

## Bug

Django's PostgreSQL backend stores `psycopg_pool.ConnectionPool` objects in a
class-level dict (`DatabaseWrapper._connection_pools`). When gunicorn (or any
pre-forking server) forks worker processes, all children inherit references to
the same pool objects — and crucially, the same underlying TCP sockets to
PostgreSQL. Multiple workers then read/write the same socket concurrently,
corrupting the PostgreSQL wire protocol.

### Observed errors

- `psycopg.DatabaseError: lost synchronization with server: got message type "…"`
- `psycopg.DatabaseError: message contents do not agree with length in message type "D"`
- `RuntimeError: generator raised StopIteration` (in Django's `from_db()`)
- `IndexError: list index out of range` (in `apply_converters`)
- `DoesNotExist` for records that exist
- Wrong types returned (e.g. `bool` where `datetime` expected)
- `ValueError: Field 'id' expected a number but got 'none'`

### Key observation

- **`-w 1` (single worker): works fine** — no fork, no shared sockets
- **`-w N` (N > 1): protocol corruption** — forked workers share TCP sockets

### Root cause

```python
# django/db/backends/postgresql/base.py
class DatabaseWrapper(BaseDatabaseWrapper):
    _connection_pools = {}   # class-level dict — survives fork()

    @property
    def pool(self):
        if self.alias not in self._connection_pools:
            pool = ConnectionPool(...)
            self._connection_pools.setdefault(self.alias, pool)
        return self._connection_pools[self.alias]
```

### Workaround

```python
# gunicorn.conf.py
def post_fork(server, worker):
    from django.db.backends.postgresql.base import DatabaseWrapper
    DatabaseWrapper._connection_pools.clear()
```

### Suggested fix

Use `os.register_at_fork(after_in_child=...)` to clear `_connection_pools`
in child processes, or check `os.getpid()` in the `pool` property and
recreate when it differs from the creating process.

## Reproduction

```bash
# 1. Create a PostgreSQL database
createdb testdb

# 2. Install dependencies
pip install django psycopg[binary] psycopg-pool gunicorn

# 3. Run migrations
python manage.py migrate

# 4. Run the test
python test_fork_bug.py
```

## Tested with

- Django 6.0.2
- psycopg 3.2.x – 3.3.2
- psycopg-pool 3.2.x – 3.3.0
- gunicorn 25.x (`--worker-class asgi`)
- Python 3.12 – 3.14
