from django.db import models
from django.utils import timezone

from project.enums.core import TargetType

__all__ = [
    "Target",
]

class Target(models.Model):
    """
    Model storing team/user metric targets.

    """

    id = models.IntegerField(primary_key=True)
    created_at = models.DateTimeField(default=timezone.now)
    updated_at = models.DateTimeField(default=timezone.now)

    key = models.TextField()
    org = models.ForeignKey("flmodels.Organization", related_name="targets", on_delete=models.DO_NOTHING)
    team = models.ForeignKey("flmodels.Team", null=True, on_delete=models.DO_NOTHING)
    apex_user = models.ForeignKey("flmodels.ApexUser", null=True, on_delete=models.DO_NOTHING)

    value = models.FloatField()
    target_type_id = models.IntegerField(choices=TargetType.choices)


follow_targets = Target.objects.filter(key="follow:max")
# output: 
# <QuerySet [<Target id=39180 key='follow:max' org=org_id name=redacted slug=redacted team=team_id name=redacted apex_user=None>, 
# <Target id=39310 key='follow:max' org=org_id name=redacted team=team_id name=redacted apex_user=None>, 
# <Target id=39291 key='follow:max' org=org_id name=redacted slug=redacted team=None apex_user=None>, 
# <Target id=39543 key='follow:max' org=org_id name=redacted team=None apex_user=None>, 
# <Target id=37798 key='follow:max' org=org_id name=redacted slug=redacted> team=team_id name=redacted> apex_user=None>, 
# '...(remaining elements truncated)...']>

org_targets = Target.objects.filter(org_id=8006)
# output:
# <QuerySet [<Target id=39180 key='follow:max' org=<Organization id=8006 name=redacted slug=redacted team=team_id name=redacted apex_user=None>, 
# <Target id=39345 key='reaction:tgt' org=<Organization id=8006 name=redacted team=team_id name=redacted apex_user=None>, 
# <Target id=39346 key='reaction:tgt' org=<Organization id=8006 name=redacted slug=redacted team=redacted name=redacted apex_user=None>, 
# <Target id=40341 key='reaction:tgt' org=<Organization id=8006 name=redacted slug=redacted team=team_id name=redacted apex_user=None>, 
# '...(remaining elements truncated)...']>


union_targets = follow_targets.union(org_targets)
print(union_targets.query)
# (
#   SELECT 
#       "core_targets"."id" AS "col1", 
#       "core_targets"."created_at" AS "col2", 
#       "core_targets"."updated_at" AS "col3", 
#       "core_targets"."key" AS "col4", 
#       "core_targets"."org_id" AS "col5", 
#       "core_targets"."team_id" AS "col6", 
#       "core_targets"."apex_user_id" AS "col7", 
#       "core_targets"."value" AS "col8", 
#       "core_targets"."target_type_id" AS "col9" 
#   FROM 
#       "core_targets" 
#   WHERE "core_targets"."key" = follow:max
# ) UNION (
#   SELECT 
#       "core_targets"."id" AS "col1", 
#       "core_targets"."created_at" AS "col2", 
#       "core_targets"."updated_at" AS "col3", 
#       "core_targets"."key" AS "col4", 
#       "core_targets"."org_id" AS "col5", 
#       "core_targets"."team_id" AS "col6", 
#       "core_targets"."apex_user_id" AS "col7",
#       "core_targets"."value" AS "col8", 
#       "core_targets"."target_type_id" AS "col9" 
#   FROM 
#       "core_targets" 
#   WHERE "core_targets"."org_id" = 8006
# )


