import logging
from datetime import date

from django.contrib.postgres.indexes import GinIndex
from django.contrib.postgres.search import SearchVectorField
from django.core import serializers
from django.db import connection, models
from django.db.models.constraints import UniqueConstraint
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.utils import timezone
from django.utils.translation import gettext as _
from mpublico.choices import *
from mpublico.tasks import (celery_crawl_empresa, celery_crawl_licitacion,
                            celery_crawl_licitacion_cambio_estado,
                            celery_crawl_licitacion_detalle,
                            celery_crawl_licitacion_especifica,
                            celery_crawl_ordencompra,
                            celery_crawl_ordencompra_cambioestado,
                            celery_crawl_ordencompra_fix,
                            celery_crawl_ordencompra_detalle)

from . import helpers
from .stop_words import STOP_WORDS

logger = logging.getLogger(__name__)

# Create your models here.


class Configuracion(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    start_crawl_date = models.DateField(default=date(2020, 1, 1))

    class Meta:
        verbose_name = 'Configuracion'
        verbose_name_plural = 'Configuracion'

    def __str__(self):
        return str(self.start_crawl_date)


class Empresa(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    CodigoEmpresa = models.CharField(_('Código empresa'), unique=True)
    NombreEmpresa = models.CharField(_('Razón social'), max_length=1000)

    class Meta:
        verbose_name = 'Empresa'
        verbose_name_plural = 'Empresa'

    class Meta:
        indexes = [
            models.Index(fields=['NombreEmpresa', 'CodigoEmpresa']),
            models.Index(fields=['NombreEmpresa']),
        ]

    def __str__(self):
        return self.NombreEmpresa


''' Dia Buscados '''


class LicitacionDia(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    empresa = models.ForeignKey(Empresa, on_delete=models.CASCADE)
    date = models.DateField(_("Fecha"))
    licitaciones_cantidad = models.IntegerField(default=0)

    class Meta:

        constraints = [UniqueConstraint(
            fields=['empresa', 'date'], name='unique_licitacion_dia')]
        verbose_name = 'Licitacion Días Buscados'
        verbose_name_plural = 'Licitacion Días Buscados'
        indexes = [
            models.Index(fields=['licitaciones_cantidad', 'date', 'empresa']),
            models.Index(fields=['licitaciones_cantidad', 'date']),
            models.Index(fields=['date', 'empresa']),
            models.Index(fields=['date']),
        ]

    def __str__(self):
        return self.empresa.NombreEmpresa


class OrdenCompraDia(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    empresa = models.ForeignKey(Empresa, on_delete=models.CASCADE)
    date = models.DateField(_("Fecha"))
    ordenescompra_cantidad = models.IntegerField(default=0)

    class Meta:

        constraints = [UniqueConstraint(
            fields=['empresa', 'date'], name='unique_ordenescompra_dia')]
        verbose_name = 'Orden Compra Días Buscados'
        verbose_name_plural = 'Orden Compra Días Buscados'
        indexes = [
            models.Index(fields=['ordenescompra_cantidad', 'date', 'empresa']),
            models.Index(fields=['ordenescompra_cantidad', 'date']),
            models.Index(fields=['date', 'empresa']),
            models.Index(fields=['date']),
        ]

    def __str__(self):
        return self.empresa.NombreEmpresa


''' Licitacion y Ordenes de Compra Base '''


class Licitacion(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    CodigoExterno = models.CharField(_('Código externo'), unique=True)
    CodigoEstado = models.IntegerField(
        _('Código estado'), choices=ESTADO_LICITACION_CHOICES, blank=True, null=True)
    FechaCierre = models.DateTimeField(
        _('Fecha Cierre'), blank=True, null=True)
    Nombre = models.CharField(_('Nombre Producto'), blank=True, null=True)
    date = models.DateField(_("Fecha Publicación"), blank=True, null=True)
    empresa = models.ForeignKey(
        Empresa, on_delete=models.CASCADE, blank=True, null=True)

    class Meta:
        indexes = [
            models.Index(
                fields=['CodigoExterno', 'CodigoEstado', 'empresa', 'date']),
            models.Index(fields=['CodigoExterno', 'CodigoEstado', 'empresa']),
            models.Index(fields=['empresa', 'date']),
            models.Index(fields=['CodigoExterno']),
            models.Index(fields=['CodigoEstado']),
            models.Index(fields=['date']),
            models.Index(fields=['empresa']),
        ]
        verbose_name = 'Licitacion Base'
        verbose_name_plural = 'Licitacion Base'

    def __str__(self):
        return self.CodigoExterno


class OrdenCompra(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    Codigo = models.CharField(_('Código externo'), unique=True)
    CodigoEstado = models.IntegerField(
        _('Código estado'), choices=ESTADO_OCOMPRA_CHOICES)
    Nombre = models.CharField(_('Nombre Producto'))
    date = models.DateField(_("Fecha Publicación"), blank=True, null=True)
    empresa = models.ForeignKey(Empresa, on_delete=models.CASCADE)

    class Meta:
        indexes = [
            models.Index(
                fields=['Codigo', 'CodigoEstado', 'empresa', 'date']),
            models.Index(fields=['Codigo', 'CodigoEstado', 'empresa']),
            models.Index(fields=['empresa', 'date']),
            models.Index(fields=['Codigo']),
            models.Index(fields=['CodigoEstado']),
            models.Index(fields=['date']),
            models.Index(fields=['empresa']),
        ]
        verbose_name = 'Orden Compra Base'
        verbose_name_plural = 'Orden Compra Base'

    def __str__(self):
        return str(self.Codigo)


''' Licitacion y Ordenes de Compra Detalle '''


class LicitacionDetalle(models.Model):
    licitacion = models.ForeignKey(
        Licitacion, on_delete=models.CASCADE, related_name='licitacion_detalle')
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    active = models.BooleanField(default=True)
    CodigoExterno = models.CharField()
    CodigoEstado = models.IntegerField(
        _('Código estado'), choices=ESTADO_LICITACION_CHOICES)
    Descripcion = models.CharField(
        _('Descripción de la Licitación u objetode la contratación'), null=True, blank=True)
    Estado = models.CharField(
        _('Estado en el que se encuentra la Licitación'), null=True, blank=True)
    Informada = models.BooleanField(
        _('Indica si la Licitación es Informada'), null=True, blank=True)
    DiasCierreLicitacion = models.IntegerField(
        _('cantidad de días para el cierre de la Licitación'), null=True, blank=True)
    CodigoTipo = models.IntegerField(
        _('Tipo Licitación'), choices=CODIGO_TIPO_LICITACION_CHOICES, null=True, blank=True)
    Tipo = models.CharField(
        _('Tipo de Licitación de Mercado Publico'), choices=TIPO_LICITACION_CHOICES, null=True, blank=True)
    TipoConvocatoria = models.IntegerField(
        _('Tipo Convocatoria'), choices=TIPO_CONVOCATORIA_CHOICES, null=True, blank=True)
    Moneda = models.CharField(
        _('Código de la moneda en la Licitación'), choices=MONEDA_CHOICES, null=True, blank=True)
    Etapas = models.IntegerField(
        _('Número de etapas de la licitación'), null=True, blank=True)
    EstadoEtapas = models.IntegerField(
        _('aperturas realizadas en la licitación'), null=True, blank=True)
    TomaRazon = models.IntegerField(
        _('Toma de Razón por Contraloría'), choices=TOMA_RAZON_LICITACION_CHOICES, null=True, blank=True)
    EstadoPublicidadOfertas = models.IntegerField(
        _('Muestra publicidad Oferta Técnica'), choices=ESTADO_PUBLICIDAD_OFERTAS_CHOICES, null=True, blank=True)
    JustificacionPublicidad = models.CharField(
        _('justifica mostrar la oferta técnica para público'), null=True, blank=True)
    Contrato = models.CharField(
        _('Tipo contrato'), choices=CONTRATO_CHOICES, null=True, blank=True)
    ObservacionContract = models.CharField(null=True, blank=True)
    Obras = models.IntegerField(
        _('Licitación es de tipo obra'), choices=OBRAS_CHOICES, null=True, blank=True)
    CantidadReclamos = models.IntegerField(
        _('Número de reclamos que se ha recibido la Institución u Organismo Público responsable de la Licitación'), null=True, blank=True)
    Tiempo = models.IntegerField(null=True, blank=True)
    UnidadTiempoEvaluacion = models.IntegerField(
        _('unidad de tiempo de la evaluación'), null=True, blank=True)
    DireccionVisita = models.CharField(
        _('Dirección de visita'), null=True, blank=True)
    DireccionEntrega = models.CharField(
        _('Dirección de entrega'), null=True, blank=True)
    Estimacion = models.IntegerField(
        _('Código de tipo de estimación'), choices=ESTIMACION_CHOICES, null=True, blank=True)
    FuenteFinanciamiento = models.CharField(
        _('Código de tipo de estimación'), null=True, blank=True)
    VisibilidadMonto = models.BooleanField(
        _('Hacer público el monto estimado en la ficha de la licitación'), null=True, blank=True)
    MontoEstimado = models.FloatField(
        _('Monto estimado que maneja el Organismo o Institución para licitar'), null=True, blank=True)
    UnidadTiempo = models.IntegerField(
        _('Informa el tipo de unidad de tiempo asociada a la etiqueta TiempoDuracionContrato'), choices=UNIDAD_TIEMPO_CHOICES, null=True, blank=True)
    Modalidad = models.IntegerField(
        _('Tipo Modalidad de pago. Plazos de Pago (30 Dias, 60 Dias, etc'), choices=MODALIDAD_CHOICES, null=True, blank=True)
    TipoPago = models.IntegerField(
        _('Indica el tipo de pago que se realizará'), null=True, blank=True)
    NombreResponsablePago = models.CharField(
        _('Nombre de la persona responsable del pago'), null=True, blank=True)
    EmailResponsablePago = models.CharField(
        _('Email responsable pago'), null=True, blank=True)
    NombreResponsableContrato = models.CharField(
        _('Nombre responsable pago'), null=True, blank=True)
    EmailResponsableContrato = models.CharField(
        _('Email responsable contrato'), null=True, blank=True)
    FonoResponsableContrato = models.CharField(
        _('Fono responsable contrato'), null=True, blank=True)
    ProhibicionContratacion = models.CharField(
        _('Prohibición Contratación'), null=True, blank=True)
    SubContratacion = models.BooleanField(
        _('Permite subcontratación'), null=True, blank=True)
    UnidadTiempoDuracionContrato = models.IntegerField(
        _('Unidad de tiempo del contrato'), choices=UNIDAD_TIEMPO_CHOICES, null=True, blank=True)
    TiempoDuracionContrato = models.IntegerField(
        _("Duración del contrato"), null=True, blank=True)
    TipoDuracionContrato = models.CharField(
        _("Nombre de período de tiempo"), null=True, blank=True)
    JustificacionMontoEstimado = models.CharField(
        _("Justificación del monto estimado"), null=True, blank=True)
    ExtensionPlazo = models.BooleanField(
        _('Si a la fecha de cierre de recepción de ofertas, se han recibido 2 o menos propuestas, el plazo de cierre se ampliará automáticamente en 2 días hábiles, por una sola vez, bajo las condiciones establecidas por el artículo 25, inciso final, del reglamento de la ley 19.886.'), null=True, blank=True)
    EsBaseTipo = models.BooleanField(
        _('licitación fue creada a través de licitaciones tipo'), null=True, blank=True)
    UnidadTiempoContratoLicitacion = models.IntegerField(
        _('Código de período de tiempo'), null=True, blank=True)
    ValorTiempoRenovacion = models.IntegerField(
        _('valor de tiempo'), null=True, blank=True)
    PeriodoTiempoRenovacion = models.CharField(
        _('Nombre del período de tiempo de renovación'), null=True, blank=True)
    EsRenovable = models.BooleanField(
        _('periodo de Renovación de Contrato es renovable'), null=True, blank=True)
    # Fechas
    Fechas_FechaCreacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaCierre = models.DateTimeField(blank=True, null=True)
    Fechas_FechaInicio = models.DateTimeField(blank=True, null=True)
    Fechas_FechaFinal = models.DateTimeField(blank=True, null=True)
    Fechas_FechaPubRespuestas = models.DateTimeField(blank=True, null=True)
    Fechas_FechaActoAperturaTecnica = models.DateTimeField(
        blank=True, null=True)
    Fechas_FechaActoAperturaEconomica = models.DateTimeField(
        blank=True, null=True)
    Fechas_FechaPublicacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaAdjudicacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaEstimadaAdjudicacion = models.DateTimeField(
        blank=True, null=True)
    Fechas_FechaSoporteFisico = models.DateTimeField(blank=True, null=True)
    Fechas_FechaTiempoEvaluacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaEstimadaFirma = models.DateTimeField(blank=True, null=True)
    Fechas_FechasUsuario = models.DateTimeField(blank=True, null=True)
    Fechas_FechaVisitaTerreno = models.DateTimeField(blank=True, null=True)
    Fechas_FechaEntregaAntecedentes = models.DateTimeField(
        blank=True, null=True)
    # Comprador
    Comprador_RutUnidad = models.CharField(blank=True, null=True)
    Comprador_CodigoUnidad = models.CharField(blank=True, null=True)
    Comprador_NombreUnidad = models.CharField(blank=True, null=True)
    Comprador_DireccionUnidad = models.CharField(blank=True, null=True)
    Comprador_ComunaUnidad = models.CharField(blank=True, null=True)
    Comprador_RegionUnidad = models.CharField(blank=True, null=True)
    Comprador_RutUsuario = models.CharField(blank=True, null=True)
    Comprador_CodigoUsuario = models.CharField(blank=True, null=True)
    Comprador_NombreUsuario = models.CharField(blank=True, null=True)
    Comprador_CargoUsuario = models.CharField(blank=True, null=True)
    Comprador_CodigoOrganismo = models.CharField(blank=True, null=True)
    Comprador_NombreOrganismo = models.CharField(blank=True, null=True)
    # Adjudicacion
    Adjudicacion_Tipo = models.IntegerField(
        choices=TIPO_ADJUDICACION_CHOICES, blank=True, null=True)
    Adjudicacion_Fecha = models.DateTimeField(blank=True, null=True)
    Adjudicacion_Numero = models.CharField(blank=True, null=True)
    Adjudicacion_NumeroOferentes = models.IntegerField(blank=True, null=True)
    Adjudicacion_UrlActa = models.URLField(
        max_length=1000, blank=True, null=True)
    # items
    Items_Cantidad = models.IntegerField(blank=True, null=True)

    def __str__(self):
        return str(self.CodigoExterno)

    class Meta:
        verbose_name = 'Licitacion Detalle'
        verbose_name_plural = 'Licitacion Detalle'

        indexes = [
            models.Index(
                fields=['MontoEstimado', 'licitacion', 'CodigoExterno', 'active']),
            models.Index(fields=['MontoEstimado', 'licitacion']),
            models.Index(fields=['MontoEstimado']),
            models.Index(fields=['Adjudicacion_UrlActa']),
            models.Index(fields=['active', 'CodigoExterno']),
            models.Index(fields=['active']),
        ]


class OrdenCompraDetalle(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    ordencompra = models.ForeignKey(
        OrdenCompra, on_delete=models.CASCADE, related_name='ordencompra_detalle')
    active = models.BooleanField(default=True)
    Codigo = models.CharField()
    CodigoEstado = models.IntegerField(
        _('Código Estado Organismo'), choices=ESTADO_OCOMPRA_CHOICES)
    Descripcion = models.CharField(
        _('Descripción de la Licitación u objetode la contratación'))
    CodigoTipo = models.IntegerField(
        _('Tipo'), choices=TIPO_OCOMPRA_CHOICES, blank=True, null=True)
    Estado = models.CharField(
        _('Estado en el que se encuentra la OC'))
    TipoMoneda = models.CharField()
    Tipo = models.CharField(choices=TIPO_ORDENCOMPRA)
    CodigoEstadoProveedor = models.IntegerField(
        _('Código Estado Proveedor'), choices=ESTADO_OCOMPRA_CHOICES)
    EstadoProveedor = models.CharField()
    Nombre = models.CharField()
    Fechas_FechaCreacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaEnvio = models.DateTimeField(blank=True, null=True)
    Fechas_FechaAceptacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaCancelacion = models.DateTimeField(blank=True, null=True)
    Fechas_FechaUltimaModificacion = models.DateTimeField(
        blank=True, null=True)
    TieneItems = models.BooleanField(_("Indica si tiene Items"), default=False)
    PromedioCalificacion = models.FloatField()
    CodigoLicitacion = models.CharField(blank=True, null=True)
    CantidadEvaluacion = models.FloatField(
        _("Evaluación del Proveedor"),)
    Descuentos = models.FloatField()
    Cargos = models.FloatField()
    TotalNeto = models.FloatField()
    PorcentajeIva = models.FloatField()
    Impuestos = models.FloatField()
    Total = models.FloatField()
    Financiamiento = models.CharField()
    Pais = models.CharField()
    TipoDespacho = models.IntegerField(
        blank=True, null=True, choices=TIPODESPACHO_OCOMPRA_CHOICES)
    FormaPago = models.IntegerField(
        blank=True, null=True, choices=TIPODESPACHO_OCOMPRA_CHOICES)
    # Comprador
    Comprador_CodigoOrganismo = models.CharField()
    Comprador_NombreOrganismo = models.CharField()
    Comprador_RutUnidad = models.CharField()
    Comprador_CodigoUnidad = models.CharField()
    Comprador_NombreUnidad = models.CharField()
    Comprador_Actividad = models.CharField(blank=True, null=True)
    Comprador_DireccionUnidad = models.CharField(blank=True, null=True)
    Comprador_ComunaUnidad = models.CharField(blank=True, null=True)
    Comprador_RegionUnidad = models.CharField()
    Comprador_Pais = models.CharField()
    Comprador_NombreContacto = models.CharField()
    Comprador_CargoContacto = models.CharField()
    Comprador_FonoContacto = models.CharField(blank=True, null=True)
    Comprador_MailContacto = models.EmailField(blank=True, null=True)
    # items
    Items_Cantidad = models.IntegerField()

    def __str__(self):
        return str(self.Codigo)

    class Meta:
        verbose_name = 'Orden Compra Detalle'
        verbose_name_plural = 'Orden Compra Detalle'

        indexes = [
            models.Index(fields=['ordencompra', 'Codigo', 'active']),
            models.Index(fields=['Codigo', 'active']),
            models.Index(fields=['Codigo']),
        ]


''' Proveedor'''


class Proveedor(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    RutProveedor = models.CharField()
    NombreProveedor = models.CharField()

    def __str__(self):
        return self.NombreProveedor

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=['RutProveedor', 'NombreProveedor'], name='unique_proovedor')
        ]
        verbose_name = 'Proveedor'
        verbose_name_plural = 'Proveedor'

        indexes = [
            models.Index(fields=['RutProveedor', 'NombreProveedor']),
            models.Index(fields=['NombreProveedor']),
        ]


class ProveedorSucursal(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    RutSucursal = models.CharField()
    CodigoSucursal = models.CharField()
    NombreSucursal = models.CharField()
    Codigo = models.CharField()
    Actividad = models.CharField(blank=True, null=True)
    Direccion = models.CharField(blank=True, null=True)
    Comuna = models.CharField(blank=True, null=True)
    Region = models.CharField(blank=True, null=True)
    Pais = models.CharField(blank=True, null=True)
    Nombre = models.CharField(blank=True, null=True)

    def __str__(self):
        return self.Nombre

    class Meta:

        verbose_name = 'Proveedor Sucursal'
        verbose_name_plural = 'Proveedor Sucursal'
        indexes = [
            models.Index(
                fields=['RutSucursal']),
        ]


''' Items '''


class LicitacionItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    licitacion = models.ForeignKey(
        Licitacion, on_delete=models.CASCADE, related_name='licitacion_items')
    licitacion_detalle = models.ForeignKey(
        LicitacionDetalle, on_delete=models.CASCADE, related_name='licitacion_detalle_items')
    producto = models.ForeignKey(
        'Producto', on_delete=models.CASCADE, blank=True, null=True)
    categoria = models.ForeignKey(
        'Categoria', on_delete=models.CASCADE, blank=True, null=True)
    proveedor = models.ForeignKey(
        Proveedor, on_delete=models.CASCADE, null=True, blank=True)
    CodigoExterno = models.CharField()
    Descripcion = models.CharField(blank=True, null=True)
    UnidadMedida = models.CharField(_('Unidad Medida'), blank=True, null=True)
    Cantidad = models.FloatField(_('Cantidad Licitada'), blank=True, null=True)
    Adjudicacion_Cantidad = models.FloatField(
        _('Cantidad Adjudicada'), blank=True, null=True)
    Adjudicacion_MontoUnitario = models.FloatField(
        _('Monto Unitario Adjudicación'), blank=True, null=True)
    active = models.BooleanField(_('Vigencia Item'), default=True)
    Correlativo = models.IntegerField(blank=True, null=True)
    analizado = models.BooleanField(default=False)
    touch = models.BooleanField(default=False)
    search = SearchVectorField(
        null=True,
        help_text="used to store all searchable info",
    )

    def __str__(self):
        return self.Descripcion

    class Meta:
        verbose_name = 'Licitacion Items Seguimiento Diario'
        verbose_name_plural = 'Licitacion Items Seguimiento Diario'
        indexes = [
            models.Index(fields=['Descripcion', 'active', 'Cantidad']),
            models.Index(fields=['Cantidad', 'licitacion']),
            models.Index(fields=['active', 'CodigoExterno']),
            models.Index(fields=['Descripcion', 'licitacion']),
            models.Index(fields=['Cantidad']),
            models.Index(fields=['Descripcion']),
            models.Index(fields=['active']),
            GinIndex(fields=["search"]),
        ]

    def update_search_vector(self):
        """
        Update SearchVector field of SearchModel using raw SQL
        search field is used to store SearchVector
        """
        db_table = self._meta.db_table

        # first get anything interesting out of the media
        # that needs to be searchable

        if self.id:

            items = []
            if self.Descripcion:
                items.append(str(self.Descripcion))

            if self.proveedor:
                if self.proveedor.NombreProveedor:
                    items.append(str(self.proveedor.NombreProveedor))

            if self.categoria:
                if self.categoria.Categoria:
                    items.append(str(self.categoria.Categoria))

            if self.licitacion:
                if self.licitacion.Nombre:
                    items.append(str(self.licitacion.Nombre))

            items = [item for item in items if item]
            text = " ".join(items)
            text = " ".join([token for token in text.lower().split(
                " ") if token not in STOP_WORDS])

            text = helpers.clean_query(text)

            sql_code = """
                UPDATE {db_table} SET search = to_tsvector(
                    '{config}', '{text}'
                ) WHERE {db_table}.id = {id}
                """.format(
                db_table=db_table, config="simple", text=text, id=self.id
            )

            try:
                with connection.cursor() as cursor:
                    cursor.execute(sql_code)
            except BaseException as e:
                logger.error(f': LicitacionItem {str(e)}')
                pass
        return True


class OrdenCompraItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    ordencompra = models.ForeignKey(
        OrdenCompra, on_delete=models.CASCADE, related_name='ordencompra_items')
    ordencompra_detalle = models.ForeignKey(
        OrdenCompraDetalle, on_delete=models.CASCADE, related_name='ORDENCOMPRA_detalle_items')
    licitacion = models.ForeignKey(
        Licitacion, on_delete=models.CASCADE, related_name='ordencompra_licitacion', blank=True, null=True)
    categoria = models.ForeignKey(
        'Categoria', on_delete=models.CASCADE, blank=True, null=True)
    producto = models.ForeignKey(
        'Producto', on_delete=models.CASCADE, blank=True, null=True)
    proveedor_sucursal = models.ForeignKey(
        ProveedorSucursal, on_delete=models.CASCADE, null=True, blank=True)
    Correlativo = models.IntegerField(blank=True, null=True)
    Producto = models.CharField(blank=True, null=True)
    EspecificacionComprador = models.CharField(null=True,
                                               blank=True)
    Moneda = models.CharField()
    Codigo = models.CharField()
    EspecificacionProveedor = models.CharField(blank=True, null=True)
    EspecificacionProveedor_normalized = models.CharField(
        blank=True, null=True)
    Cantidad = models.FloatField()
    Unidad = models.CharField(blank=True, null=True)
    PrecioNeto = models.FloatField()
    TotalCargos = models.FloatField(blank=True, null=True)
    TotalDescuentos = models.FloatField(blank=True, null=True)
    TotalImpuestos = models.FloatField(blank=True, null=True)
    Total = models.FloatField(blank=True, null=True)
    active = models.BooleanField(_('Vigencia Item'), default=True)
    analizado = models.BooleanField(default=False)
    touch = models.BooleanField(default=False)
    search = SearchVectorField(
        null=True,
        help_text="used to store all searchable info",
    )

    def __str__(self):
        return str(self.EspecificacionComprador)

    class Meta:
        verbose_name = 'Orden Compra Items Seguimiento Diario'
        verbose_name_plural = 'Orden Compra Items Seguimiento Diario'
        indexes = [
            models.Index(fields=['Producto', 'active', 'Cantidad']),
            models.Index(fields=['Cantidad', 'ordencompra']),
            models.Index(fields=['Producto', 'ordencompra']),
            models.Index(fields=['Cantidad']),
            models.Index(fields=['Producto']),
            models.Index(fields=['active']),
            GinIndex(fields=["search"]),
        ]

    def update_search_vector(self):
        """
        Update SearchVector field of SearchModel using raw SQL
        search field is used to store SearchVector
        """
        db_table = self._meta.db_table

        # first get anything interesting out of the media
        # that needs to be searchable

        if self.id:

            items = []
            if self.EspecificacionComprador:
                items.append(str(self.EspecificacionComprador))

            if self.EspecificacionProveedor:
                items.append(str(self.EspecificacionProveedor))

            if self.categoria:
                if self.categoria.Categoria:
                    items.append(str(self.categoria.Categoria))

            if self.Producto:
                items.append(str(self.Producto))

            if self.proveedor_sucursal:
                if self.proveedor_sucursal.Nombre:
                    items.append(str(self.proveedor_sucursal.Nombre))

            items = [item for item in items if item]
            text = " ".join(items)
            text = " ".join([token for token in text.lower().split(
                " ") if token not in STOP_WORDS])

            text = helpers.clean_query(text)

            sql_code = """
                UPDATE {db_table} SET search = to_tsvector(
                    '{config}', '{text}'
                ) WHERE {db_table}.id = {id}
                """.format(
                db_table=db_table, config="simple", text=text, id=self.id
            )

            try:
                with connection.cursor() as cursor:
                    cursor.execute(sql_code)
            except BaseException as e:
                logger.error(f': LicitacionItem {str(e)}')
                pass
        return True


''' Comunes '''


class Categoria(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    Categoria = models.CharField()
    CodigoCategoria = models.CharField()

    def __str__(self):
        return str(self.Categoria)

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=['Categoria', 'CodigoCategoria'], name='unique_categoria')
        ]
        verbose_name = 'Categoria'
        verbose_name_plural = 'Categoria'

        indexes = [
            models.Index(fields=['CodigoCategoria', 'Categoria']),
            models.Index(fields=['Categoria']),
        ]


class Producto(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    NombreProducto = models.CharField()
    CodigoProducto = models.CharField()
    categoria = models.ForeignKey(
        Categoria, on_delete=models.CASCADE, blank=True, null=True)

    def __str__(self):
        return self.NombreProducto

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=['NombreProducto', 'CodigoProducto'], name='unique_producto')
        ]

        verbose_name = 'Producto'
        verbose_name_plural = 'Producto'

        indexes = [
            models.Index(fields=['NombreProducto',
                         'categoria', 'CodigoProducto']),
            models.Index(fields=['NombreProducto', 'categoria']),
            models.Index(fields=['NombreProducto']),
        ]


class Error(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    host = models.CharField(blank=True, null=True)
    error_msge = models.CharField(blank=True, null=True)
    stage = models.CharField(blank=True, null=True)

    def __str__(self):
        return self.error_msge

    class Meta:
        verbose_name = 'Error'
        verbose_name_plural = 'Errores'


class Adquisicion(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    file = models.FileField(upload_to="uploads/%Y/%m/%d/")
    crawl = models.CharField()
    model = models.CharField()

    class Meta:
        verbose_name = 'Adquisicion'
        verbose_name_plural = 'Adquisicion'

    def __str__(self):
        return self.crawl


''' ZGEN CENABAST '''


class Zgen(models.Model):
    zgen = models.CharField()
    nombre_generico = models.CharField()
    descripcion = models.CharField()
    tipo = models.CharField()

    def __str__(self):
        return str(self.zgen)

    class Meta:
        verbose_name = 'Zgen CENABAST'
        verbose_name_plural = 'Zgen CENABAST'


class ZgenClasificacionOrdenCompraItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    item = models.ForeignKey(OrdenCompraItem, on_delete=models.CASCADE,
                             related_name='ordencompra_item_zgen_clasificacion')
    result = models.ForeignKey(
        Zgen, on_delete=models.CASCADE, related_name='result_zgen_orden_compra')

    def __str__(self):
        return str(self.result)

    class Meta:
        verbose_name = 'Orden Compra Zgen Clasificacion'
        verbose_name_plural = 'Orden Compra Zgen Clasificacion'


class ZgenClasificacionLicitacionItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    item = models.ForeignKey(
        LicitacionItem, on_delete=models.CASCADE, related_name='licitacion_item_zgen_clasificacion')
    result = models.ForeignKey(
        Zgen, on_delete=models.CASCADE, related_name='result_zgen_licitacion')

    def __str__(self):
        return str(self.result)

    class Meta:
        verbose_name = 'Licitacion Zgen Clasificacion'
        verbose_name_plural = 'Licitacion Zgen Clasificacion'


''' Principios Activos '''


class PrincipioActivo(models.Model):
    principio_activo = models.CharField(default='-')
    concentracion = models.CharField(default='-')

    def __str__(self):
        return str(self.principio_activo + ' ' + str(self.concentracion))


class PrincipioActivosLicitacionItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    item = models.ForeignKey(
        LicitacionItem, on_delete=models.CASCADE, related_name='licitacion_item_principio_activo')
    principios_activos = models.ForeignKey(
        PrincipioActivo, on_delete=models.CASCADE, related_name='licitacion_principios_activos')

    def __str__(self):
        return str(self.item)

    class Meta:
        verbose_name = 'Licitacion Principios Acivos'
        verbose_name_plural = 'Licitacion Principios Activos'


class PrincipioActivosOrdenCompraItem(models.Model):
    created_at = models.DateTimeField(
        _('Fecha'), default=timezone.now, blank=True)
    updated_at = models.DateTimeField(auto_now=True, null=True)
    item = models.ForeignKey(
        OrdenCompraItem, on_delete=models.CASCADE, related_name='ordencompra_item_principio_activo')
    principios_activos = models.ForeignKey(
        PrincipioActivo, on_delete=models.CASCADE, related_name='ordencompra_principios_activos')

    def __str__(self):
        return str(self.item)

    class Meta:
        verbose_name = 'Orden Compra Principios Acivos'
        verbose_name_plural = 'Orden Compra Principios Activos'


''' Signals '''


@receiver(post_save, sender=Adquisicion)
def crawl_proccess(sender, instance, created, **kwargs):
    logger.info('NUEVO CRAWL')
    if created:

        if instance.model == 'Empresa':
            serialized_instance = serializers.serialize('json', [instance, ])
            celery_crawl_empresa.apply_async(
                args=[serialized_instance], kwargs={})

        elif instance.model == 'Licitacion':
            celery_crawl_licitacion.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'LicitacionDetalle':
            celery_crawl_licitacion_detalle.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'LicitacionesCambioEstado':
            celery_crawl_licitacion_cambio_estado.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'LicitacionesEspecificas':
            celery_crawl_licitacion_especifica.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'LicitacionesUpdate':
            celery_crawl_licitacion_especifica.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'OrdenCompra':
            celery_crawl_ordencompra.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'OrdenCompraDetalle':
            celery_crawl_ordencompra_detalle.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'OrdenesCompraCambioEstado':
            celery_crawl_ordencompra_cambioestado.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'OrdenesCompraEspecificas':
            celery_crawl_ordencompra_detalle.apply_async(
                args=[str(instance.file)], kwargs={})

        elif instance.model == 'OrdenesCompraFix':
            ''' Este metodo se creo para verificar aquellas ordenes de compra sin productos ni categorias'''
            ''' Pruebas indican que noe s necesrio, ya que se estan capturando OK'''
            celery_crawl_ordencompra_fix.apply_async(
                args=[str(instance.file)], kwargs={})

    return instance


@receiver(post_save, sender=LicitacionItem)
def licitaciones_items_proccess(sender, instance, created, **kwargs):
    instance.update_search_vector()
    return instance


@receiver(post_save, sender=OrdenCompraItem)
def ordencompra_items_proccess(sender, instance, created, **kwargs):
    instance.update_search_vector()
    return instance
