from common.db.basic import Logged
from django.db import models
from django.utils.translation import gettext_lazy as _
from django_fsm import FSMField, transition

from shipment.models.trip import Trip


class ShipmentStateChoices(models.TextChoices):
    NEW = "NW", _("new")  # before the trip is created
    WAITING = "WA", _("waiting for courier")  # Trip is created but no courier has been assigned to it yet
    PICKUP = "PI", _(
        "on the source way")  # Trip is assigned to a courier and the courier is on his way to pickup packages from the source
    DROP_OFF = "DO", _(
        "delivering to destination")  # Trip is picked up by the courier, and he is on his way to deliver the packages
    DELIVERED = "DV", _("delivered")
    CANCELLED = "CA", _("cancelled")


class ShipmentQuerySet(models.QuerySet):
    def actives(self, *args, **kwargs):
        return super(ShipmentQuerySet, self).filter(*args, **kwargs).filter(
            state__in=[
                ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING, 
                ShipmentStateChoices.PICKUP, ShipmentStateChoices.DROP_OFF
                ])

    def closes(self, *args, **kwargs):
        return super(ShipmentQuerySet, self).filter(*args, **kwargs).filter(
            state__in=[
                ShipmentStateChoices.DELIVERED, ShipmentStateChoices.CANCELLED
            ])

class ShipmentManager(models.Manager):
    def get_queryset(self):
        return ShipmentQuerySet(self.model, using=self._db)

    def actives(self, *args, **kwargs):
        return self.get_queryset().actives(*args, **kwargs)

    def closes(self, *args, **kwargs):
        return self.get_queryset().closes(*args, **kwargs)


class Shipment(Logged):
    order_id = models.PositiveIntegerField(verbose_name=_('order id'))
    trip = models.ForeignKey(Trip, on_delete=models.PROTECT, verbose_name=_('trip'), null=True,
                             related_name='shipments')
    state = FSMField(default=ShipmentStateChoices.NEW, choices=ShipmentStateChoices.choices,
                     verbose_name=_('State'), protected=True)
    tracking_number = models.CharField(verbose_name=_('tracking number'), max_length=400, null=True, blank=True)
    tracking_url = models.CharField(verbose_name=_('tracking URL'), null=True, blank=True, max_length=500)
    total_weight = models.FloatField(verbose_name=_('total weight'), null=True, blank=True)
    destination = models.JSONField(verbose_name=_('destination address'))
    dropped_off_at = models.DateTimeField(verbose_name=_('drop off time'), null=True, blank=True, help_text=_(
        'The exact time this course we delivered to the customer. It will be null if the course is not delivered yet'))

    shipment_start_time = models.DateTimeField(verbose_name=_('shipment start time'), null=True, blank=True)
    shipment_end_time = models.DateTimeField(verbose_name=_('shipment end time'), null=True, blank=True)

    objects = ShipmentManager()

    class Meta:
        verbose_name = _('Shipment')
        verbose_name_plural = _("Shipments")
        unique_together = ('order_id', 'trip')
        index_together = [
            ['order_id', 'trip']
        ]

    @property
    def is_cancellable(self):
        return self.state in [ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING, ShipmentStateChoices.PICKUP]

    @property
    def is_editable(self):
        return self.state in [ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING]

    @transition(field='state', source=[
        ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING, ShipmentStateChoices.PICKUP,
        ShipmentStateChoices.DROP_OFF], target=ShipmentStateChoices.WAITING,
                custom={'button_name': _('set as waiting')})
    def set_as_waiting(self):
        pass

    @transition(field='state', source=[
        ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING, ShipmentStateChoices.PICKUP,
        ShipmentStateChoices.DROP_OFF
    ], target=ShipmentStateChoices.PICKUP,
                custom={'button_name': _('set as pickup')})
    def set_as_pickup(self):
        pass

    @transition(field='state', source=[
        ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING, ShipmentStateChoices.PICKUP,
        ShipmentStateChoices.DROP_OFF], target=ShipmentStateChoices.DROP_OFF,
                custom={'button_name': _('set as drop off')})
    def set_as_drop_off(self):
        pass

    @transition(field='state', source=[
        ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING,
        ShipmentStateChoices.PICKUP, ShipmentStateChoices.DROP_OFF
    ], target=ShipmentStateChoices.CANCELLED,
                custom={'button_name': _('set as cancelled')})
    def set_as_cancelled(self):
        if self.is_cancellable:
            handler = self.trip.handler
            if self.trip.shipments.all().count() == 1:
                response = handler.cancel_trip(self.trip)
            else:
                response = handler.remove_course(course=self)
            return response
        else:
            raise Exception(_("Can not cancel the shipment. The courier is on the way!"))

    @transition(field='state', source=[
        ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING,
        ShipmentStateChoices.PICKUP, ShipmentStateChoices.DROP_OFF, ShipmentStateChoices.DELIVERED],
                target=ShipmentStateChoices.DELIVERED,
                custom={'button_name': _('set as delivered')})
    def set_as_delivered(self):
        pass

    # This is for admin panel
    @transition(field='state', source=[ShipmentStateChoices.NEW, ShipmentStateChoices.WAITING,
                                       ShipmentStateChoices.PICKUP, ShipmentStateChoices.DROP_OFF],
                target=ShipmentStateChoices.DELIVERED,
                custom={'button_name': _('delivered')})
    def delivered(self):
        pass

    # def update_instance(self):
    #     raise NotImplementedError
    #

    @classmethod
    def check_active_shipment_exists(cls, order_id):
        return cls.objects.filter(order_id=order_id).exclude(
            state__in=[ShipmentStateChoices.CANCELLED, ShipmentStateChoices.DELIVERED]
        ).exists()

    def update_driver_info(self, driver_info):
        new_trip = Trip.objects.create(
            source=self.trip.source,
            method=self.trip.method,
            driver_info=driver_info,
            driver=driver_info['national_code']
        )
        old_trip = self.trip
        self.trip = new_trip
        self.save()
        if not old_trip.shipments.exists():
            old_trip.delete()


class ShipmentItem(Logged):
    shipment = models.ForeignKey(Shipment, on_delete=models.CASCADE, verbose_name=_('Shipment'))
    state = FSMField(default=ShipmentStateChoices.NEW, choices=ShipmentStateChoices.choices,
                     verbose_name=_('State'))
    order_item_id = models.PositiveIntegerField(verbose_name=_('order Item id'))
    title = models.CharField(verbose_name=_('title'), max_length=200)
    quantity = models.FloatField(verbose_name=_("Quantity"))
    warehouse_id = models.PositiveIntegerField(verbose_name=_("warehouse id"), null=True)

    class Meta:
        verbose_name = _('Shipment')
        verbose_name_plural = _("Shipment")
