from common.db.basic import Logged
from django.core.exceptions import ValidationError
from django.core.validators import MaxValueValidator
from django.db import models
from django.utils.translation import gettext_lazy as _

from shipment.models.method import ShippingMethod, ShippingChoice


class Trip(Logged):
    trip_id = models.CharField(verbose_name=_('trip id'), max_length=400, null=True, blank=True)
    method = models.ForeignKey(ShippingMethod, on_delete=models.PROTECT, verbose_name=_('method'))
    source = models.JSONField(verbose_name=_('source address'))
    trip_cost = models.PositiveIntegerField(verbose_name=_('trip cost'), null=True, blank=True, default=0)
    driver_info = models.JSONField(verbose_name=_('driver info'), null=True, blank=True, )
    driver = models.PositiveBigIntegerField(
        verbose_name=_("driver national code"), validators=[MaxValueValidator(9999999999)], null=True, blank=True
    )

    class Meta:
        verbose_name = _('Trip')
        verbose_name_plural = _("Trips")
        unique_together = ('trip_id', 'method')
        index_together = ('trip_id', 'method')

    @property
    def handler(self):
        return self.method.handler

    def __str__(self):
        return str(self.trip_id)

    def cancel(self):
        for shipment in self.shipments.all():
            shipment.set_as_cancelled()

    def clean(self):
        if self.method.title == ShippingChoice.ORGANIZATION:
            if self.driver is None:
                raise ValidationError({'driver': _('driver is empty')})
        else:
            if self.trip_id is None:
                raise ValidationError({'trip_id': _('trip_id is empty')})
        super(Trip, self).clean()
