from common.db.basic import Logged, Ordered
from django.db import models
from django.utils.translation import gettext_lazy as _

from tehran_plus_common.models.base import StoreTypeChoices


class ShippingChoice(models.TextChoices):
    MIARE = 'Miare', _('Miare')
    ALOPEYK = 'Alopeyk', _('Alopeyk')
    ORGANIZATION = 'Organization', _('Organization')  # like motorbikes for grocery or MRF vehicles


class ShippingMethodQuerySet(models.QuerySet):
    def actives(self, *args, **kwargs):
        return super(ShippingMethodQuerySet, self).filter(*args, **kwargs).filter(is_active=True)


class ShippingMethodManager(models.Manager):
    def get_queryset(self):
        return ShippingMethodQuerySet(self.model, using=self._db)

    def actives(self, *args, **kwargs):
        return self.get_queryset().actives(*args, **kwargs)


class ShippingMethod(Logged, Ordered):
    title = models.CharField(verbose_name=_("title"), max_length=200, choices=ShippingChoice.choices, unique=True)
    description = models.TextField(verbose_name=_("description"), blank=True, null=True)
    is_active = models.BooleanField(verbose_name=_('is active'), default=True)
    max_course_in_trip = models.PositiveIntegerField(verbose_name=_('max number of courses in one trip'))

    class Meta:
        verbose_name = _('Shipping Method')
        verbose_name_plural = _("Shipping Methods")

    def __str__(self):
        return self.title

    @classmethod
    def is_method(cls, title):
        return cls.objects.filter(title=title).exists()

    @property
    def handler(self):
        from ..miare_controller import MiareShipmentHandler
        from ..organization_controller import OrganizationShipmentHandler

        if self.title == ShippingChoice.MIARE.value:
            return MiareShipmentHandler()
        elif self.title == ShippingChoice.ORGANIZATION.value:
            return OrganizationShipmentHandler()
        return None  # should never be returned!

    objects = ShippingMethodManager()


class ShippingMethodStoreType(models.Model):
    store_type = models.PositiveSmallIntegerField(verbose_name=_("store type"), choices=StoreTypeChoices.choices)
    shipping_method = models.ForeignKey(ShippingMethod, on_delete=models.CASCADE, verbose_name=_('shipping method'))
    is_available = models.BooleanField(default=True, verbose_name=_('is_available'))

    class Meta:
        verbose_name = _('Shipping Method Store Type')
        verbose_name_plural = _("Shipping Methods Stores Types")
