from django.test.client import Client
import unittest
import settings

class TestPostGetPost(unittest.TestCase):

    def echo_view(self, request):
        # Django view, gets monkey-patched in to urls.py
        from django.http import HttpResponse
        response = HttpResponse()
        response.write(request.method + '\n')
        keys = request.POST.keys()
        keys.sort()
        response.write(','.join(keys))
        return response

    def setUp(self):
        # Monkey-patch in our test view
        urls = __import__(settings.ROOT_URLCONF, {}, {}, [''])
        import copy
        urls.oldpatterns = copy.copy(urls.urlpatterns)
        urls.urlpatterns += urls.patterns('', 
            ('^echomethod/$', self.echo_view)
        )

    def tearDown(self):
        urls = __import__(settings.ROOT_URLCONF, {}, {}, [''])
        urls.urlpatterns = urls.oldpatterns
        del urls.oldpatterns

    def test_get(self):
        response = Client().get('/echomethod/')
        self.assert_(response.content.startswith('GET'))

    def test_post(self):
        response = Client().post('/echomethod/', {'foo': 'bar'})
        self.assertEqual(response.content, 'POST\nfoo')

    def test_post_multiple_fields(self):
        response = Client().post('/echomethod/', {'foo': 'bar', 'baz': '1'})
        self.assertEqual(response.content, 'POST\nbaz,foo')

    def test_get_post_get(self):
        client = Client()
        
        response = client.get('/echomethod/')
        self.assert_(response.content.startswith('GET'))
        
        response = client.post('/echomethod/', {'foo': 'bar', 'baz': '1'})
        self.assertEqual(response.content, 'POST\nbaz,foo')
        
        response = client.get('/echomethod/')
        self.assert_(response.content.startswith('GET'))

    def test_post_get_post_not_unicode(self):
        client = Client()
        data = {
            'firstname': 'Test',
            'lastname': 'Example',
            'email': 'test@example.com',
        }
        keys = data.keys()
        keys.sort()

        response = client.post('/echomethod/', data)
        self.assertEqual(response.content, 'POST\n%s' % ','.join(keys))

        response = client.get('/echomethod/')
        self.assert_(response.content.startswith('GET'))

        response = client.post('/echomethod/', data)
        self.assertEqual(response.content, 'POST\n%s' % ','.join(keys))

    def test_post_get_post_unicode(self):
        client = Client()
        data = {
            u'firstname': u'Test',
            u'lastname': u'Example',
            u'email': u'test@example.com',
        }
        keys = data.keys()
        keys.sort()
        
        response = client.post('/echomethod/', data)
        self.assertEqual(response.content, 'POST\n%s' % ','.join(keys))

        response = client.get('/echomethod/')
        self.assert_(response.content.startswith('GET'))

        response = client.post('/echomethod/', data)
        self.assertEqual(response.content, 'POST\n%s' % ','.join(keys))

if __name__ == '__main__':
    unittest.main()