from django.db import models

class Company(models.Model):
    name = models.TextField(max_length=500, unique=True, verbose_name='Company Name')
    locations = models.ManyToManyField(Location, through="CompanyLocation")
    record_status = models.TextField(max_length=25, unique=True, verbose_name='Record Status')

class Location(models.Model):
    name = models.TextField(max_length=500, unique=True, verbose_name='Location Name')
    record_status = models.TextField(max_length=25, unique=True, verbose_name='Record Status')

class CompanyLocation(models.Model):
    company = models.ForeignKey(Company, on_delete=models.CASCADE)
    location = models.ForeignKey(
        Location, on_delete=models.CASCADE, related_name="companies"
    )
    record_status = models.TextField(max_length=25, unique=True, verbose_name='Record Status')

    class Meta:
        unique_together = ["company", "location"]


# QuerySet w/ Prefetch Example:
queryset = (
	models.Company.objects
	.prefetch_related(
		# Only active Locations
		Prefetch(
			"locations",
			queryset=models.Location.filter(record_status="Active")
			.extra(where=["demo_companylocation.record_status='Active'"]),
		),
	)
	.all()
	# Only active Companies
	.filter(record_status="Active")
	.order_by("name")
)