from django.contrib import admin
from .models import Question, Choice
from .utils import collect_gateway_context
from .views import payment_view



class QuestionAdmin(admin.ModelAdmin):

    model = Question

    def change_view(self, request, object_id, form_url="", extra_context=None):
        """
        self = QuestionAdmin
        object_id = question_id
        form_url is passed through to self.render_change_form() via
        self.get_preserved_filters(request) and then added to context
        extra_context = dict of apps, models, admin_urls and permissions.
        Ensure self.change_form_template = None so next time it doesn't use payment.html
        """
        extra_context = extra_context or {}
        question = Question.objects.get(id=object_id)
        self.change_form_template = None
        if not question.token:
            self.change_form_template = "payment.html"
            context = collect_gateway_context(question)
            extra_context.update(self.admin_site.each_context(request))
            extra_context.update(context)
            self.admin_site.admin_view(
                payment_view(request, question, context=extra_context)
            )
        return super().change_view(request, object_id, form_url, extra_context)


    fieldsets = (
        (
            None,
            {
                "description": "",
                "classes": ("wide",),
                "fields": (
                    "token",
                    "question_text",
                    "pub_date",
                ),
            },
        ),
    )

    class ChoiceInline(admin.StackedInline):

        model = Choice
        extra = 0
        fk_name = "question"
        fieldsets = (
            (
                "Choices",
                {
                    "description": "",
                    "classes": ("wide",),
                    "fields": (
                        "choice_text",
                        "votes",
                    ),
                },
            ),
        )

    inlines = [
        ChoiceInline,
    ]

admin.site.register(Question, QuestionAdmin)
