from django.conf import settings
from django.contrib import admin
from django.contrib.auth.admin import UserAdmin
from django.http import HttpRequest
from django.http.response import HttpResponseRedirect
from django.utils import translation
from django.utils.translation import gettext_lazy as _

from .models import User


@admin.register(User)
class CustomUserAdmin(UserAdmin):
    fieldsets = (
        (None, {"fields": ("username", "password")}),
        (_("Personal info"), {"fields": ("first_name", "last_name", "email")}),
        (_("Settings"), {"fields": ("language",)}),
        (_("Permissions"), {"fields": ("is_active", "is_staff", "is_superuser", "groups", "user_permissions")}),
        (_("Important dates"), {"fields": ("last_login", "date_joined")}),
    )

    def response_post_save_change(  # type: ignore[override]
        self, request: HttpRequest, obj: User
    ) -> HttpResponseRedirect:
        """
        Applies user language right after user edit.

        :param request:
        :param obj:
        :return:
        """
        response = super().response_post_save_change(request, obj)

        # Only if user is self-editing
        if request.user == obj:
            translation.activate(obj.language)
            response.set_cookie(
                settings.LANGUAGE_COOKIE_NAME, obj.language, secure=settings.LANGUAGE_COOKIE_SECURE, samesite="lax"
            )

        return response
