import sys
import timeit
from collections import namedtuple

from django.utils.functional import SimpleLazyObject

ALREADY_PROFILING = 'cProfile' in sys.modules
if not ALREADY_PROFILING:
    import cProfile
import os
from django.conf import settings
import django
from django.template import Template, Context

ROOT = os.path.dirname(django.__file__)

def urls():
    from django.urls import path
    from django.contrib import admin
    return (
        path('admin', admin.site.urls),
    )

settings.configure(
    DEBUG=False,
    INSTALLED_APPS=('django.contrib.contenttypes', 'django.contrib.auth', 'django.contrib.admin',),
    ROOT_URLCONF=SimpleLazyObject(urls),
    TEMPLATES=[
        {
            "BACKEND": "django.template.backends.django.DjangoTemplates",
            "DIRS": [],
            "APP_DIRS": True,
            "OPTIONS": {
                "context_processors": [
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
                ],
            },
        },
    ],
)
django.setup()


changelist_path = os.path.join(ROOT, 'contrib', 'admin', 'templates', 'admin', 'change_list.html')
with open(changelist_path, 'r') as f:
    changelist = f.read()

templates = {
    'admin_changelist': Template(changelist),
    'single_textnode': Template(
    """
<!doctype html>
    <html class="no-js" lang="">

    <head>
      <meta charset="utf-8">
      <title></title>
      <meta name="description" content="">
      <meta name="viewport" content="width=device-width, initial-scale=1">

      <meta property="og:title" content="">
      <meta property="og:type" content="">
      <meta property="og:url" content="">
      <meta property="og:image" content="">

      <link rel="manifest" href="site.webmanifest">
      <link rel="apple-touch-icon" href="icon.png">
      <!-- Place favicon.ico in the root directory -->

      <link rel="stylesheet" href="css/normalize.css">
      <link rel="stylesheet" href="css/style.css">

      <meta name="theme-color" content="#fafafa">
    </head>
    <body>

      <!-- Add your site or application content here -->
      <p>Hello world! This is HTML5 Boilerplate.</p>
      <script src="js/vendor/modernizr-3.11.7.min.js"></script>
      <script src="js/app.js"></script>
      <!-- Google Analytics: change UA-XXXXX-Y to be your site's ID. -->
      <script>
        window.ga = function () { ga.q.push(arguments) }; ga.q = []; ga.l = +new Date;
        ga('create', 'UA-XXXXX-Y', 'auto'); ga('set', 'anonymizeIp', true); ga('set', 'transport', 'beacon'); ga('send', 'pageview')
      </script>
      <script src="https://www.google-analytics.com/analytics.js" async></script>
    </body>
    </html>
    """
),
}
fakeopts = namedtuple('Opts', 'app_label object_name')('auth', 'user')
class ChangeList:
    result_count = 0
    full_result_count = 0
    opts = fakeopts
    list_display = ()
    formset = None
    result_list = ()
    show_all = False
    multi_page = False
    can_show_all = True
    def get_ordering_field_columns(self):
        return []
ctx = Context({
    'cl': ChangeList(),
    'opts': fakeopts
})

if __name__ == "__main__":
    args_count = len(sys.argv)
    runner = 'profile'
    template = 'single_textnode'
    if args_count > 1:
        runner = sys.argv[1]
    if args_count > 2:
        template = sys.argv[2]

    if template not in templates:
        sys.stderr.write("Expected one of\n{0!s}\nfor template name\n".format(", ".join(templates)))
        sys.exit(1)

    if runner == 'profile':
        sys.stdout.write("Running cProfile for\n{0!s}\n".format(template))

        if not ALREADY_PROFILING:
            sys.stdout.write("Starting cProfile".format(template))
            pr = cProfile.Profile()
            pr.enable()

        def profile_many():
            t = templates[template]
            for x in range(100_000):
                t.render(ctx)

        profile_many()

        if not ALREADY_PROFILING:
            pr.disable()
            sys.stdout.write("Stopping cProfile".format(template))
            pr.print_stats()

    elif runner == 'timeit':
        sys.stdout.write("Running timeit for\n{0!s}\n".format(template))

        def profile():
            t = templates[template]
            t.render(ctx)

        out = timeit.timeit(profile, number=100_000)
        sys.stdout.write("100000 loops -> {secs:.{prec}g} secs\n".format(
            secs=out,
            prec=4,
        ))

    elif runner == 'yappi':
        import yappi
        sys.stdout.write("Running yappi for\n{0!s}\n".format(template))

        yappi.start()

        def profile_many():
            t = templates[template]
            for x in range(100_000):
                t.render(ctx)

        profile_many()

        yappi.stop()
        yappi.get_func_stats().print_all()

    else:
        sys.stderr.write("Expected one of\nprofile, timeit, yappi\nfor execution method\n")
        sys.exit(1)


