'use strict';

import {obj_remove_id, sortByKey, urlEncode} from "./library";
import Cookie from 'js-cookie';


class RestAPI {
    static get BASE_URL() {
        return '/api/';
    }

    static get SAFE_METHODS() {
        return ['GET', 'HEAD', 'OPTIONS', 'TRACE'];
    }

    static get EXPECTED_CODE() {
        return {
            DELETE: 204,
            POST: 201,
        };
    }

    static get JSON_TYPES() {
        return ['application/json'];
    }

    static request(method, resource, data = null, headers = null) {
        method = method.toUpperCase();

        let url = `${this.BASE_URL}${resource}/`;

        if (data !== null && this.SAFE_METHODS.includes(method)) {
            // need to add parameters to URL
            url += '?' + urlEncode(data);
        }

        return fetch(url, {
            method: method,
            credentials: 'same-origin',
            headers: {...this.get_headers(method), ...headers},
            body: this.get_body(method, data)
        }).then(response => {
            if (!response.ok) {
                return Promise.reject([`Failed to ${method} ${resource}`, response]);
            }

            let expected_code = this.EXPECTED_CODE[method];
            if (expected_code && response.status !== expected_code) {
                return Promise.reject([`Invalid response code: ${response.status} != ${expected_code}`, response]);
            }

            let content_type = response.headers.get('Content-Type');

            if (content_type && this.JSON_TYPES.includes(content_type)) {
                // return JSON only when we know that response is JSON (e.g. not empty response)
                return response.json();
            }

            return response.text().then(text => text || null);
        });
    }

    static get(resource, data, headers) {
        return this.request('GET', resource, data, headers);
    }

    static post(resource, data, headers) {
        return this.request('POST', resource, data, headers);
    }

    static patch(resource, data, headers) {
        return this.request('PATCH', resource, data, headers);
    }

    static options(resource, data, headers) {
        return this.request('OPTIONS', resource, data, headers);
    }

    static delete(resource, data, headers) {
        return this.request('DELETE', resource, data, headers);
    }

    static get_headers(method) {
        if (this.SAFE_METHODS.includes(method)) {
            return {};
        }
        return {
            'X-CSRFToken': Cookie.get('csrftoken'),
            'Content-Type': 'application/json; charset="utf-8"',
        };
    }

    static get_body(method, data) {
        if (data === null || this.SAFE_METHODS.includes(method)) {
            return null;
        }
        return JSON.stringify(data);
    }
}


export class API {
    static get DOMAIN() {
        throw Error('You must define DOMAIN');
    }

    static get ORDER_BY() {
        return 'name';
    }

    static create(obj) {
        return RestAPI.post(this.DOMAIN, obj_remove_id(obj));
    }

    static all() {
        return RestAPI.get(this.DOMAIN).then(objects => objects.sort(sortByKey(this.ORDER_BY)));
    }

    static get(obj) {
        return RestAPI.get(`${this.DOMAIN}/${obj.id}`);
    }

    static update(obj) {
        return RestAPI.patch(`${this.DOMAIN}/${obj.id}`, obj);
    }

    static delete(obj) {
        return RestAPI.delete(`${this.DOMAIN}/${obj.id}`);
    }

    static options(headers) {
        return RestAPI.options(this.DOMAIN, null, headers);
    }

    static extract_schema(metadata) {
        return metadata.actions.POST;
    }
}
