from django.core.urlresolvers import reverse
import sys
import django
from django.test import TransactionTestCase
from django.views.debug import ExceptionReporter
from django.contrib.auth import views
from django.contrib.auth.forms import AuthenticationForm

try:
    from unittest.mock import patch
except:
    from mock import patch


class TestClientNotRaisingExceptionButCapturing(object):

    def __init__(self, client):
        self.client = client
        self.stored_exc_info = None
        self.forcing_test_client_to_ignore_exception = patch.object(
            self.client, 'store_exc_info',
            side_effect=self.capture_exc_info_instead_of_client
        )
        self.mock_client_store_exc_info = None

    def __enter__(self):
        self.mock_client_store_exc_info = \
            self.forcing_test_client_to_ignore_exception.__enter__()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        return self.mock_client_store_exc_info.__exit__(
            exc_type, exc_val, exc_tb)

    def get_captured_request(self):
        if hasattr(self.mock_client_store_exc_info, 'assert_called_once'):
            self.mock_client_store_exc_info.assert_called_once()
        else:
            assert 1 == self.mock_client_store_exc_info.call_count, dict(
                expected=1, actual=self.mock_client_store_exc_info.call_count)
        args, kwargs = self.mock_client_store_exc_info.call_args
        return kwargs['request']

    def capture_exc_info_instead_of_client(self, *args, **kwargs):
        self.stored_exc_info = sys.exc_info()


class ReproTestCase(TransactionTestCase):

    def test_when_login_view_raises_an_exception_password_is_not_in_the_500_email(self):  # noqa: E501
        password = '$0m3 P4$$w0rd'
        exception_email_html_body = self.get_500_email_html_for_login_error(
            username='admin/some_user', password=password
        )
        self.assertNotIn(
                member=password, container=exception_email_html_body)

###

    def get_500_email_html_for_login_error(self, username, password):
        # patch this methodd so AuthenticationForm.clean is
        # called which has local password variable
        login_view_raising_value_error = patch(
            'django.contrib.auth.forms.authenticate',
            side_effect=ValueError('some error')
        )

        self.goto_login_page()

        with TestClientNotRaisingExceptionButCapturing(self.client) as capture:
            with login_view_raising_value_error:
                self.submit_login(username=username, password=password)

        request = capture.get_captured_request()
        exc_type, exc_value, tb = capture.stored_exc_info
        # based on django.utils.log.AdminEmailHandler.emit
        reporter = ExceptionReporter(
            request=request, is_email=True,
            exc_type=exc_type, exc_value=exc_value, tb=tb)
        self.assertTrue(reporter.filter.is_active(request))
        return reporter.get_traceback_html()

    def goto_login_page(self, **query):
        url = reverse('login')
        self.login_get_response = self.client.get(url)
        self.assertEqual(200, self.login_get_response.status_code)

    def submit_login(self, username, password, **query):
        url = reverse('login')
        self.login_post_response = self.client.post(
            url, dict(username=username, password=password))
