from django.db import models
from django.contrib.auth.models import User


class Customer(models.Model):
	"""
	classe che gestische le informazioni relativi al cliente
	"""
	
	name = models.CharField(max_length = 255, null = False, blank = False)
	description = models.TextField(max_length = 5000, null = True, blank = True)
	enabled = models.BooleanField(null = False, blank = False, default = True)
	
	def __str__(self):
		return self.name
	
class CustomerUser(User):
	"""
	Estensione dell'utente base per memorizzare i dati dell'utente della mailing list

	"""
	customer = models.ForeignKey(Customer, blank = False, null = True)
	
	
class Domain(models.Model):
		"""
		Classe che contiene e gestisce l'elenco dei domini gestiti dal tool
		di mailing list
		"""
		
		domain = models.CharField(max_length = 100, null = False, blank = False)
		name = models.CharField(max_length = 255, null = False, blank = False)
		description = models.TextField(max_length = 5000, null = True, blank = True)
		code = models.CharField(max_length = 50, null = False, blank = False)
		full_domain = models.CharField(max_length = 255, null = False, blank = False, default = "---")
		catch_all_forward = models.EmailField(null = True, blank = True)
		
		customer = models.ForeignKey(Customer, blank = True, null = True)
		
		def __str__(self):
			return self.domain


class Channel(models.Model):
	"""
	Classe che contiene e gestisce i dati dei canali di invio di ogni cliente
	"""
	
	name = models.CharField(max_length = 255, null = False, blank = False)
	description = models.TextField(max_length = 5000, null = True, blank = True)
	enabled = models.BooleanField(null = False, blank = False, default = True)
	code = models.CharField(max_length = 50, null = True, blank = True)
	
	domain = models.ForeignKey(Domain, null = False, blank = False)
	
	def __str__(self):
		return self.name
	
class Template(models.Model):
	"""
	Classe che contiene l'elenco dei templati
	Un invio può essere associato solo a un template
	"""
	name = models.CharField(max_length = 255, null = False, blank = False)
	link = models.URLField(null =True, blank = True)
	path = models.FileField(null = True, blank = True, upload_to = "mail_templates")
	code = models.CharField(max_length = 50, null = True, blank = True)
	
	def __str__(self):
		return self.name
	
	
class Shipment(models.Model):
	"""
	Classe che traccia i dati degli invii
	"""
	
	date = models.DateField(null = False, blank = False)
	description = models.TextField(	max_length = 5000, null = True, blank = True)
	code = models.CharField(max_length = 50, null = True, blank = True)
	subject = models.CharField(max_length = 100, null = True, blank = True)
	sent = models.BooleanField(blank = False, null = False, default = False)
	started = models.DateTimeField(blank = True, null = True)
	finished = models.DateTimeField(blank = True, null = True)
	
	channel = models.ForeignKey(Channel, null = False, blank = False)
	template = models.ForeignKey(Template, null = False, blank = False)
	
	def __str__(self):
		return "Shipment of " + str(self.date)
	
	
class Recipient(models.Model):
	"""
	Classe che contiene e gestisce l'elenco dei destinatari
	"""
	
	mail = models.EmailField(null = False, blank = False)
	fullname = models.CharField(max_length = 500, null = True, blank = True)
	enabled = models.BooleanField(null = False, blank = False, default = True)
	unsubscribed = models.BooleanField(null = False, blank = False, default = False)
	unsubscribe_date = models.DateTimeField(null = True, blank = True)
	
	channel = models.ForeignKey(Channel, null = False, blank = False)
	shipments = models.ManyToManyField(Shipment, blank = True)
	
	def __str__(self):
		return self.mail + (self.fullname if self.fullname is not None else "")

	
class ShipmentData(models.Model):
	"""
	Classe che memorizza tutti i dati dell'invio sotto forma di chiave/valore
	collegati a un destinatario
	"""
	
	field = models.CharField(max_length = 25, null = False, blank = False)
	value = models.TextField(max_length = 10000, null = False, blank = True)
	deleted = models.BooleanField(null = False, blank = False, default = False)
	
	shipment = models.ForeignKey(Shipment, null = False, blank = False)
	recipient = models.ForeignKey(Recipient, null = False, blank = False)
	
	def __str__(self):
		return self.recipient.mail + " - " + self.field


class ChannelData(models.Model):
	"""
	Classe che memorizza tutti i dati del canale sotto forma di chiave/valore
	collegati a un destinatario
	"""
	
	field = models.CharField(max_length = 25, null = False, blank = False)
	value = models.TextField(max_length = 10000, null = False, blank = True)
	deleted = models.BooleanField(null = False, blank = False, default = False)
	
	channel = models.ForeignKey(Channel, null = False, blank = False)
	recipient = models.ForeignKey(Recipient, null = False, blank = False)
	
	def __str__(self):
		return self.recipient.mail + " - " +self.field
	
	
class Placeholder(models.Model):
	"""
	Classe che contiene l'elenco dei placeholder ricavati da un template
	"""

	name = models.CharField(max_length = 100, null = False, blank = False)
	description = models.TextField(max_length = 5000, null = True, blank = True)
	
	template = models.ForeignKey(Template, null = False, blank = False)
	
	def __str__(self):
		return self.name
	
	
class Link(models.Model):
	"""
	Classe con l'elenco dei link di un template
	"""
	
	name = models.CharField(max_length = 100, null = True, blank = True, default = "")
	original = models.CharField(max_length = 200, null = False, blank = False)
	category = models.CharField(max_length = 100, null = True, blank = True, default = "")
	destination = models.CharField(max_length = 100, null = False, blank = False)
	
	template = models.ForeignKey(Template, null = False, blank = False, related_name = "link_template")
	
	def __str__(self):
		return self.destination
	
	
class LogClick(models.Model):
	"""
	Classe di memorizzazione dei click sui link
	"""
	
	click_date = models.DateField(null = False, blank = False)
	
	link = models.ForeignKey(Link, null = True, blank = True)
	recipient = models.ForeignKey(Recipient, null = True, blank = True)
	shipment = models.ForeignKey(Shipment, null = True, blank = True)
	
	
class LogDelivery(models.Model):
	"""
	Classe che memorizza i dati di ogni singolo invio per ogni destinatario
	"""
	
	delivery_date = models.DateTimeField(null = False, blank = False)
	message_id = models.CharField(max_length = 50, null = False, blank = False)
	bounce_receive_date = models.DateField(null = True, blank = True)
	bounce_dsn_code = models.CharField(max_length = 20, null = True, blank = True)
	bounce_action = models.CharField(max_length = 50, null = True, blank = True)
	bounce_remote_mta = models.CharField(max_length = 200, null = True, blank = True)
	
	recipient = models.ForeignKey(Recipient, blank = False, null = False)
	shipment = models.ForeignKey(Shipment, blank = False, null =False)
	

	