"au ""tests/test_prefetch.py"""
from django.db.models.query import Prefetch
from django.test import TestCase
from .models import Author, Book, Publisher


def flatten(ls):
    return list([i for s in ls for i in s])


class PrefetchTestCase(TestCase):

    def setUp(self):
        """Create a simple fixture:

        B0 --> P0 <--> A0
        """
        publisher = Publisher.objects.create(name='Publisher0')
        Book.objects.create(name='Book0', publisher=publisher)
        author = Author.objects.create(name='Author0')
        author.publishers.add(publisher)

    def test_prefetch_nested(self):
        """
        Demonstrates the issue, fails in Django 1.10 (master), 1.9, and 1.8.
        """

        # Construct a nested prefetch queryset
        # Note: Publisher appears twice, once as the primary model and once as
        # a prefetched relationship.
        publishers = Publisher.objects.prefetch_related(
            Prefetch(
                'books',
                Book.objects.all().prefetch_related(
                    Prefetch(
                        'publisher',
                        Publisher.objects.all().prefetch_related('authors')
                    )
                )
            )
        )
        # Evaluate queryset
        # Expect 4 queries: Publishers, Books, Publishers, Authors
        with self.assertNumQueries(4):
            publishers = list(publishers)

        """
        Queries made so far:

        1)

        SELECT
            "tests_publisher"."id",
            "tests_publisher"."name"
        FROM "tests_publisher"

        2)

        SELECT
            "tests_book"."id",
            "tests_book"."name",
            "tests_book"."publisher_id"
        FROM "tests_book" WHERE "tests_book"."publisher_id" IN (1)

        3)

        SELECT
            "tests_publisher"."id",
            "tests_publisher"."name"
        FROM "tests_publisher"
        WHERE "tests_publisher"."id" IN (1)

        SELECT
            ("tests_author_publishers"."publisher_id")
                AS "_prefetch_related_val_publisher_id",
            "tests_author"."id",
            "tests_author"."name"
        FROM "tests_author"
        INNER JOIN "tests_author_publishers"
            ON ("tests_author"."id" = "tests_author_publishers"."author_id")
        WHERE "tests_author_publishers"."publisher_id" IN (1)']
        """

        # At this point, expect 0 queries as everything should be prefetched
        with self.assertNumQueries(0):
            books = flatten([p.books.all() for p in publishers])
        with self.assertNumQueries(0):
            publishers = [b.publisher for b in books]
        with self.assertNumQueries(0):
            """
            This fails with:

            AssertionError: 1 queries executed, 0 expected
            Captured queries were:

            SELECT
                "tests_author"."id",
                "tests_author"."name"
            FROM "tests_author"
            INNER JOIN "tests_author_publishers"
                ON ("tests_author"."id" = "tests_author_publishers"."author_id")
            WHERE "tests_author_publishers"."publisher_id" = 1
            """
            authors = flatten([p.authors.all() for p in publishers])  # noqa

    def test_prefetch_flat(self):
        """
        Analogous case using flat prefetch structure that succeeds
        """

        # Construct a flat prefetch queryset
        # Note: Publisher appears twice, once as the primary model and once as
        # a prefetched relationship.
        publishers = Publisher.objects.prefetch_related(
            'books__publisher__authors'
        )

        # Evaluate queryset
        # Expect 3 queries: Publishers, Books, Authors
        # Note: the flat prefetch form optimizes out the 4th query
        # against publishers, given that they must have been seen already.
        with self.assertNumQueries(3):
            publishers = list(publishers)

        """
        Queries made so far:

        1)

        SELECT
            "tests_publisher"."id",
            "tests_publisher"."name"
        FROM "tests_publisher"

        2)

        SELECT
            "tests_book"."id",
            "tests_book"."name",
            "tests_book"."publisher_id"
        FROM "tests_book"
        WHERE "tests_book"."publisher_id" IN (1)

        3)

        SELECT
            ("tests_author_publishers"."publisher_id")
                AS "_prefetch_related_val_publisher_id",
            "tests_author"."id",
            "tests_author"."name"
        FROM "tests_author"
        INNER JOIN "tests_author_publishers"
            ON ("tests_author"."id" = "tests_author_publishers"."author_id")
        WHERE "tests_author_publishers"."publisher_id" IN (1)
        """

        # At this point, expect 0 queries as everything should be prefetched
        with self.assertNumQueries(0):
            books = flatten([p.books.all() for p in publishers])
        with self.assertNumQueries(0):
            publishers = [b.publisher for b in books]
        with self.assertNumQueries(0):
            # No problem here!
            authors = flatten([p.authors.all() for p in publishers])  # noqa
