class Transaction(models.Model):
    """
    Financial tracking entry. Either expense, income or transfer between two
    accounts.

    user - owner of the transaction
    date - date when transaction occurred
    type - one of 3: 'EXPENSES', 'INCOME' or 'TRANSFER'
    amount - amount of money in transaction
    currency - currency that was used for the transaction
    amount_primary - amount in user's primary currency
    category and sub_category - ids for category and sub_category if the are
        present
    account - account that was used in transaction
    source_account - if transfer, money go from it to `account`
    tags - another way to categorize transactions. Transaction may only have
        one category but as many tags as needed
    project - yet another way to categorize transactions and share them (TODO)
    note - optional user's note
    recurrence - repeat config for repeating transactions
    recurrence_parent - link to original transaction for repeating
        transactions
    exchange_rate - json data
    updated_at - when transaction was edited last time
    created_at - when transaction was created
    """

    EXPENSES, INCOME, TRANSFER = 'EXPENSES', 'INCOME', 'TRANSFER'
    TRANSACTION_TYPE_CHOICES = (
        (EXPENSES, EXPENSES),
        (INCOME, INCOME),
        (TRANSFER, TRANSFER),
    )

    user = models.ForeignKey(User)
    date = models.DateTimeField(db_index=True)
    type = models.CharField(
        choices=TRANSACTION_TYPE_CHOICES,
        default=EXPENSES,
        max_length=10,
        db_index=True
    )
    amount = models.DecimalField(max_digits=12, decimal_places=2)
    currency = models.ForeignKey(Currency)
    amount_primary = models.DecimalField(max_digits=12, decimal_places=2,
                                         null=True, blank=True)
    category = models.ForeignKey(
        Category,
        null=True,
        blank=True,
        on_delete=models.SET_NULL
    )
    sub_category = models.ForeignKey(SubCategory, null=True, blank=True,
                                     on_delete=models.SET_NULL)
    source_account = models.ForeignKey(
        Account,
        null=True,
        blank=True,
        related_name='source_transactions',
        on_delete=models.SET_NULL
    )
    account = models.ForeignKey(Account, related_name='account_transactions')
    tags = models.ManyToManyField('Tag', null=True, blank=True,
                                  related_name='transactions')
    project = models.ForeignKey(Project, blank=True, null=True)
    note = models.TextField(null=True, blank=True)
    recurrence = models.ForeignKey('Recurrence', null=True, blank=True,
                                   on_delete=models.SET_NULL)
    recurrence_parent = models.ForeignKey(
        'Transaction',
        null=True,
        blank=True,
        on_delete=models.SET_NULL
    )
    exchange_rate = models.ForeignKey('ExchangeRate')
    updated_at = models.DateTimeField(
        auto_now=True,
        null=True,
        blank=True,
        db_index=True
    )
    created_at = models.DateTimeField(
        auto_now_add=True,
        blank=True,
        null=True,
        db_index=True
    )

    _original_amount = None
    _original_currency = None

    class Meta:
        ordering = ('-date', '-id')

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._original_amount = self.amount
        self._original_currency = getattr(self, 'currency', None)

    def save(self, *args, **kwargs):
        """
        Check if amount or currency have changed and recalculate
        primary_amount if that's the case.

        Check if sub_category and category fields don't match and
        fix it (with priority to sub_category)
        """
        if (
            self.amount_primary is None or
            self.amount != self._original_amount or
            self.currency != self._original_currency
        ):
            self.amount_primary = self.primary_amount()

        if self.sub_category is not None:
            if self.category != self.sub_category.category:
                self.category = self.sub_category.category

        super().save(*args, **kwargs)
        self._original_amount = self.amount
        self._original_currency = self.currency

    def clean(self):
        """
        Override Transaction's clean method to do validation regarding
        source_account
        """
        super(Transaction, self).clean()

        if self.type == Transaction.TRANSFER and self.source_account is None:
            raise exceptions.ValidationError({
                'source_account': ['This has to be set when type == ' +
                                   str(self.TRANSFER)]
            })
        elif self.source_account == self.account:
            raise exceptions.ValidationError({
                'source_account': ['This cannot be the same as account']
            })

        # Check that all items associated with the transaction belong to the
        # user associated aith the transaction
        if self.sub_category and self.sub_category.user != self.user:
            raise exceptions.ValidationError({
                'sub_category': ['belongs to different user']
            })

        if self.source_account and self.source_account.user != self.user:
            raise exceptions.ValidationError({
                'source_account': ['belongs to different user']
            })

        if self.account and self.account.user != self.user:
            raise exceptions.ValidationError({
                'account': ['belongs to different user']
            })

        if self.project and self.project.user != self.user:
            raise exceptions.ValidationError({
                'project': ['belongs to different user']
            })

        if self.category and self.category.user != self.user:
            raise exceptions.ValidationError({
                'category': ['belongs to different user']
            })

        # Set the project to standard project if
        # user does not explicitly set it
        if not self.project:
            project = Project.objects.filter(user=self.user,
                                             default=True).first()
            if not project:
                raise exceptions.ValidationError('Missing default project')

            self.project = project

    def primary_amount(self):
        """
        Return the user's amount in his/her primary currency
        """
        if self.user.preferences.primary_currency is None:
            return decimal.Decimal(0)

        if self.currency.code == self.user.preferences.primary_currency.code:
            return decimal.Decimal(self.amount)

        current_rate = decimal.Decimal(
                                self.exchange_rate.rates[self.currency.code]
                            )

        amount = decimal.Decimal(self.amount)

        current_usd_amount = amount/current_rate

        currency_code = self.user.preferences.primary_currency.code

        return current_usd_amount * decimal.Decimal(
                                        self.exchange_rate.rates[currency_code]
                                                   )

    def primary_currency(self):
        """
        Return the user's primary currency
        """
        return self.user.preferences.primary_currency

    def __str__(self):
        return self.user.get_full_name() + ' : ' + str(self.amount)

    @property
    def is_income(self, ):
        """
        Is this transaction an income?
        """
        return self.type == self.INCOME

    @property
    def is_expense(self, ):
        """
        Is this an expense?
        """
        return self.type == self.EXPENSES

    @property
    def is_transfer(self, ):
        """
        Is this transaction a transfer?
        """
        return self.type == self.TRANSFER
