from django.contrib.auth.models import User
from django.db import models


class Serving(models.Model):
    SERVING_CHOICES = (
        (0, "unknown"),
        (1, "tap"),
        (2, "bottle"),
        (4, "can"),
        (3, "growler"),
    )
    amount = models.DecimalField(max_digits=4, decimal_places=1)
    date = models.DateField()
    type = models.SmallIntegerField(choices=SERVING_CHOICES, default=1)
    owner = models.ForeignKey(User)
    location = models.CharField(max_length=100, blank=True)
    cost = models.DecimalField(max_digits=5,
                               decimal_places=2, null=True, blank=True)

    def __str__(self):
        return " ".join([str(self.amount), "oz of", self.beer.name])


class Week(models.Func):
    function = 'strftime'
    template = "%(function)s('%%W',%(expressions)s)"


def beer_data_last_n_weeks(user, week_count):
    assert week_count > 0, "week_count must be greater than 0"
    assert week_count < 52, "week_count must be less than 52"
    from datetime import date, timedelta
    today = date.today()
    most_recent_monday = today - timedelta(days=(today.isoweekday()-1))
    start_date = most_recent_monday - timedelta(days=7*(week_count-1))

    beer_data = user.serving_set.filter(date__range=(start_date, today)
            ).annotate(week=Week('date', output_field=models.IntegerField())).values(
            'week').annotate(oz=models.Sum('amount'), cost=models.Sum('cost'))

    return beer_data
