from __future__ import unicode_literals
from django.db import models
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User, Group
from django.utils import timezone


class ContractorManager(models.Manager):
    def has_report_due_in_month(self, month):
        from abas.apps.inspections.models import InspectionItem
        items = InspectionItem.objects.has_report_due_in_month(month) \
            .filter(is_active=True).values_list('pk', flat=True)
        contractors = Contractor.objects.filter(inspectionitem__in=(set(items))) \
            .order_by('signoff_method') \
            .distinct()
        return contractors


class Contractor(models.Model):
    """ Contractor profiles.

    @todo This shares quite a lot of information with Client. Think about maybe adding a Profile model.
    """
    SERVICE_TYPE_CHOICES = (
        ('0', 'Multipurpose'),
        ('1', 'Fire equipment'),
        ('2', 'Electrical'),
        ('3', 'Mechanicals'),
        ('4', 'Fire equipment and electrical'),
        ('5', 'Lifts'),
        ('6', 'Brigade connections'),
        ('7', 'N/A'),
        ('8', 'Passives (inc. doors & paths)'),
        ('9', 'Anchor points'),
        ('10', 'Backflow valves'),
        ('11', 'Evacuation diagrams & training'),
    )
    SIGNOFF_METHOD_CHOICES = (
        ('ONLINE', 'Online signoff module'),
        ('EMAIL', 'Email notification'),
        ('PHONE', 'Contact via phone'),
        ('SPECIAL', 'Special attention required'),
        ('NONE', 'Not necessary'),
    )
    SIGNOFF_CONTACT_CHOICES = (
        ('CONTRACTOR', 'Contractor'),
        ('CLIENT', 'Client'),
    )
    SIGNOFF_GROUPING_CHOICES = (
        ('CONTACT', 'Email per contact'),
        ('PROPERTY', 'Email per property'),
    )
    created = models.DateTimeField(auto_now_add=True, null=True)
    updated = models.DateTimeField(auto_now=True)

    name = models.CharField(max_length=100)
    account = models.ForeignKey(User, null=True, blank=True)

    primary_contact = models.CharField(max_length=100, blank=True, null=True)
    email = models.EmailField(max_length=254, blank=True, null=True)

    is_active = models.BooleanField(default=True, verbose_name="Active")

    contact_mobile = models.CharField(max_length=100, blank=True, null=True, verbose_name="Contact Mobile")
    contact_bh = models.CharField(max_length=100, blank=True, null=True, verbose_name="Contact BH")
    contact_ah = models.CharField(max_length=100, blank=True, null=True, verbose_name="Contact AH")
    contact_fax = models.CharField(max_length=100, blank=True, null=True, verbose_name="Contact Fax")

    notes = models.TextField(blank=True, null=True, help_text="Miscellaneous information about the contractor.")

    address = models.TextField(blank=True, null=True)

    service_type = models.CharField(null=True, blank=True, max_length=200, choices=SERVICE_TYPE_CHOICES)

    signoff_method = models.CharField(max_length=20, default="SIGNOFF", help_text="How should signoff requests be issued?", choices=SIGNOFF_METHOD_CHOICES)
    signoff_contact = models.CharField(max_length=20, default="CONTRACTOR", help_text="Should the contractor or client be contacted for signoff?", choices=SIGNOFF_CONTACT_CHOICES)
    signoff_grouping = models.CharField(max_length=20, default="PROPERTY", help_text="Should the contact receive a single email with all properties, or an email for each property?", choices=SIGNOFF_GROUPING_CHOICES)
    signoff_instructions = models.TextField(blank=True, null=True, help_text="Instructions to appear on the online signoff module page.")
    signoff_template = models.ForeignKey('SignoffTemplate', null=True, help_text="Template to use for signoffs.")
    signoff_email_cc = models.TextField(blank=True, null=True, help_text="Additional email addresses for signoff correspondence, one per line.")

    objects = ContractorManager()

    def get_absolute_url(self):
        return reverse('contractors:contractor_detail', kwargs={'pk': self.id})

    def create_stub_account(self):
        account = User.objects.create_user(self.email, self.email)
        account.save()
        Group.objects.get(name='Contractors').user_set.add(account)
        return account

    def __unicode__(self):
        return self.name

    class Meta:
        ordering = ['name']


class Signoff(models.Model):
    """ Contractor signoff. """
    STATUS_CHOICES = (
        ('NEW', 'New'),
        ('ACCESSED', 'Accessed'),
        ('SUBMITTED', 'Submitted'),
        ('COMPLETED', 'Completed'),
        ('ARCHIVED', 'Archived'),
    )
    created = models.DateTimeField(auto_now_add=True, null=True)
    updated = models.DateTimeField(auto_now=True)
    submitted = models.DateTimeField(blank=True, null=True, editable=False)

    period_start = models.DateField()
    period_finish = models.DateField()

    instructions = models.TextField(blank=True, null=True, help_text="Custom instructions for signoff.")

    contractor = models.ForeignKey(Contractor, help_text="The contractor object that owns these inspection items.")
    account = models.ForeignKey(User, help_text="The account assigned to complete this inspection, e.g. an account beloning to a Contractor or a Client.")

    status = models.CharField(choices=STATUS_CHOICES, default='NEW', max_length=15)

    note = models.TextField(blank=True, null=True, help_text='Administrative note for this signoff.')

    def get_loginurl(self):
        from loginurl.models import Key
        login_url = reverse('contractors:signoff_update', kwargs={ 'pk': self.pk })
        keys = Key.objects.filter(user=self.account, next=login_url)
        if keys.count():
            login_url = reverse('loginurl', kwargs={ 'key': keys[0].key })
        return login_url

    def get_items(self):
        pks = self.signoffproperty_set.exclude(is_disclaimed=True).values_list('pk', flat=True)
        return SignoffItem.objects.filter(signoff_property__in=pks)

    def get_incomplete_items(self):
        return self.get_items().filter(new_is_compliant__isnull=True)

    def get_complete_items(self):
        return self.get_items().filter(new_is_compliant__isnull=False)

    # This is used on on the /thanks/ view.
    def is_fully_completed(self):
        return True if self.get_incomplete_items().count() == 0 else False

    def get_absolute_url(self):
        return reverse('contractors:signoff_update', kwargs={ 'pk': self.pk })

    def has_access(self, user):
        # Only available to the contractor it was sent to
        # This method is not called for admins
        if self.account != user:
            return False

        # If it's archived, you can't edit it.
        if self.status == 'ARCHIVED':
            return False

        # If it's older than 45 days, you can't edit it
        age = timezone.now() - self.created
        if age.days > 45:
            return False

        # If they complete the submission, they only have 24 hours.
        if self.submitted:
            age = timezone.now() - self.submitted
            if self.status == 'COMPLETED' and age.total_seconds() > 24*60*60:
                return False

        return True

    def __unicode__(self):
        return "%s (%s to %s) [%s]" % (self.contractor.name, self.period_start, self.period_finish, self.status)


class SignoffProperty(models.Model):
    """ Contractor signoff. """
    signoff = models.ForeignKey(Signoff)
    property = models.ForeignKey('properties.Property')
    note = models.TextField(blank=True, null=True, help_text='Additional information e.g. missing information or additional items that you service for this property.')
    is_disclaimed = models.BooleanField(default=False, verbose_name="Disclaimed", help_text='If you are no longer servicing this property, use this button to notify us.')

    def disclaim_property(self):
        from .emails import signoff_disclaimed_email
        for signoffitem in self.signoffitem_set.all():
            signoffitem.disclaim_item()
        msg = signoff_disclaimed_email(self)
        msg.send()

    def get_absolute_url(self):
        return reverse('contractors:signoff_update', kwargs={ 'pk': self.signoff_id }) + '#%s' % self.pk

    # This is used on on the /thanks/ view.
    def is_fully_completed(self):
        if self.is_disclaimed:
            return True
        if self.signoffitem_set.filter(new_is_compliant__isnull=True).count() == 0:
            return True
        return False

    def __unicode__(self):
        return self.property.name

    class Meta:
        verbose_name_plural = "signoff properties"
        ordering = ['-id']


class SignoffItem(models.Model):
    """ Contractor signoff. """
    created = models.DateTimeField(auto_now_add=True, null=True)
    updated = models.DateTimeField(auto_now=True)

    signoff_property = models.ForeignKey(SignoffProperty)
    item = models.ForeignKey('inspections.InspectionItem')
    label = models.CharField(blank=True, null=True, max_length=200)

    signoff_date = models.DateField(blank=True, null=True)
    serviced_date = models.DateField(blank=True, null=True)
    signoff_note = models.TextField(blank=True, null=True)
    contractor_note = models.TextField(blank=True, null=True)
    is_compliant = models.NullBooleanField()

    new_serviced_date = models.DateField(blank=True, null=True, verbose_name='Last service date')
    new_signoff_note = models.TextField(blank=True, null=True, help_text='Non-compliance information for display on our reports.')
    new_is_compliant = models.NullBooleanField()

    def disclaim_item(self):
        # @todo this is ugly AESG specific stuff .. move somewhere.
        from abas.apps.inspections.models import InspectionItemFailure
        failure_nocontractor = InspectionItemFailure.objects.get(label='No contractor')
        contractor_needcontractor = Contractor.objects.get(name='Need contractor')

        internal_note = "%s reported they were not servicing this item on %s via the online signoff module." \
            % (self.item.contractor.name, timezone.now().strftime('%Y-%m-%d'))
        self.item.compliant = False
        self.item.signoff_date = timezone.now()
        self.item.signoff_method = 'WEBSITE'
        self.item.internal_note = "\n\n".join([internal_note, self.item.internal_note]) if self.item.internal_note else internal_note
        self.item.contractor = contractor_needcontractor
        self.item.failures.add(failure_nocontractor)
        self.item.save()


class SignoffTemplate(models.Model):
    """ Contractor per-property settings. """
    is_active = models.BooleanField(default=True)
    name = models.CharField(max_length=50, help_text="Name this template.")
    subject = models.CharField(max_length=200, help_text="The subject of the email.")
    body_plain = models.TextField(help_text="The plain text content of the email to be sent to the client or contractor.")
    body_html = models.TextField(help_text="The HTML content of the email to be sent to the client or contractor.")

    def __unicode__(self):
        return self.name


# class ContractorPropertySetting(models.Model):
#     """ Contractor per-property settings. """
#     contractor = models.ForeignKey(Contractor)
#     property = models.ForeignKey(Property)
#     general_note = models.TextField(blank=True, null=True)
#     internal_note = models.TextField(blank=True, null=True)
