# -*- coding: utf-8 -*-
from django.db import models
from django.conf import settings
from django.contrib.auth.models import (BaseUserManager, AbstractBaseUser,
    PermissionsMixin)


class BaseModel(models.Model):
    created_at = models.DateTimeField(auto_now_add=True)
    created_by = models.ForeignKey(
        settings.AUTH_USER_MODEL, null=True, blank=True, editable=False,
        related_name='%(app_label)s_%(class)s_created_related'
    )
    updated_at = models.DateTimeField(auto_now=True)
    updated_by = models.ForeignKey(
        settings.AUTH_USER_MODEL, null=True,
        blank=True, editable=False,
        related_name='%(app_label)s_%(class)s_updated_related'
    )

    class Meta:
        abstract = True


class AccountManager(BaseUserManager):
    def _create_user(self, username, email, password, is_staff, is_superuser,
        **kwargs):
        if not username:
            raise ValueError('Accounts must have a username.')
        if not email:
            raise ValueError('Accounts must have an e-mail address.')

        email = self.normalize_email(email)
        user = self.model(username=username, email=email, is_staff=is_staff,
            is_superuser=is_superuser, **kwargs)

        user.set_password(password)
        user.save(using=self._db)

        return user

    def create_user(self, username, email, password=None, **kwargs):
        return self._create_user(username, email, password, is_staff=False,
            is_superuser=False, **kwargs)

    def create_superuser(self, username, email, password, **kwargs):
        return self._create_user(username, email, password, is_staff=True,
            is_superuser=True, **kwargs)


class Account(AbstractBaseUser, PermissionsMixin, BaseModel):
    username = models.CharField(max_length=100, unique=True)
    email = models.EmailField(max_length=254, unique=True)
    first_name = models.CharField(max_length=255, blank=True)
    last_name = models.CharField(max_length=255, blank=True)
    is_staff = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)

    objects = AccountManager()

    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = ['email']

    def get_full_name(self):
        return '{0} {1}'.format(self.first_name, self.last_name)

    def get_short_name(self):
        return self.first_name

    def __str__(self):
        return self.username

    def email_user(self, subject, message, from_email=None, **kwargs):
        raise NotImplementedError()

