# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.contrib import auth
from django.contrib.auth.models import (
    AbstractBaseUser,
    AbstractUser,
    PermissionsMixin,
    BaseUserManager
    )
from django.core.mail import send_mail
from django.db import models
from django.utils import timezone
from django.utils.http import urlquote
from django.utils.translation import ugettext_lazy as _


################ Authentication ##################

class UserManager(BaseUserManager):
    def _create_user(self, email, password,
                     is_staff, is_superuser, **extra_fields):
        """
        Creates and saves a User with the given email and password.
        """
        now = timezone.now()
        if not email:
            raise ValueError('The given email must be set')
        user = self.model(email=self.normalize_email(email),
                          is_staff=is_staff, is_active=True,
                          is_superuser=is_superuser, last_login=now,
                          date_joined=now, **extra_fields)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_user(self, email, password=None, **extra_fields):
        return self._create_user(email, password, False, False,
                                 **extra_fields)

    def create_superuser(self, email, password, **extra_fields):
        return self._create_user(email, password, True, True,
                                 **extra_fields)


# Shortcuts to "inherit" base model translation strings
AbstractUserMeta = AbstractUser.__dict__['_meta']
AbstractUserFields = AbstractUserMeta.fields
AbstractUserM2MFields = AbstractUserMeta.many_to_many

class User(AbstractBaseUser, PermissionsMixin):
    """
    User model with admin-compliant permissions.

    Password and email are required. Other fields are optional.
    """
    email = models.EmailField(AbstractUserFields[6].verbose_name,
                              max_length=255, db_index=True, unique=True)
    full_name = models.CharField(_('full name'),
                                 max_length=255, blank=True)
    is_staff = models.BooleanField(AbstractUserFields[7].verbose_name,
                                   default=False,
                                   help_text=AbstractUserFields[7].help_text)
    is_active = models.BooleanField(AbstractUserFields[8].verbose_name,
                                    default=True,
                                    help_text=AbstractUserFields[8].help_text)
    date_joined = models.DateTimeField(AbstractUserFields[9].verbose_name,
                                       default=timezone.now)

    objects = UserManager()

    USERNAME_FIELD = 'email'
    REQUIRED_FIELDS = ['full_name']

    class Meta:
        verbose_name = AbstractUserMeta.verbose_name
        verbose_name_plural = AbstractUserMeta.verbose_name_plural

    def get_full_name(self):
        return self.full_name.strip()

    def get_short_name(self):
        "Returns the short name for the user."
        return self.full_name.strip()

    def email_user(self, subject, message, from_email=None, **kwargs):
        """
        Sends an email to this User.
        """
        send_mail(subject, message, from_email, [self.email], **kwargs)
