import timeit
import sys

memoize_setup = """
from django.utils.functional import memoize
def fib(n):
    if n < 2:
        return n
    return fib(n-1) + fib(n-2)
_fib = {}
fib = memoize(fib, _fib, 1)
fib(100)
"""
lru_cache_setup = """
from django.utils.lru_cache import lru_cache
@lru_cache(maxsize=None)
def fib(n):
    if n < 2:
        return n
    return fib(n-1) + fib(n-2)
fib(100)
"""

# read performance -- the cache is warmed up, so read a single entry
read_loop = """
fib(50)
"""

# write performance -- write 100 items per run, not one item per run to have
# the cache contain multiple entries, not just one which might skew the
# results.
memoize_write_loop = """
_fib.clear()
fib(100)
"""
lru_cache_write_loop = """
fib.cache_clear()
fib(100)
"""


memoize_read_time = timeit.timeit(read_loop, memoize_setup, number=10 ** 6)
lru_cache_read_time = timeit.timeit(read_loop, lru_cache_setup, number=10 ** 6)
memoize_write_time = timeit.timeit(memoize_write_loop, memoize_setup,
                                   number=10 ** 4)
lru_cache_write_time = timeit.timeit(lru_cache_write_loop, lru_cache_setup,
                                     number=10 ** 4)

print("""Python: %s
+-----------+----------+-----------+
|           | memoize  | lru_cache |
+-----------+----------+-----------+
| 1M reads  | %000.5fs | %000.5fs  |
| 1M writes | %000.5fs | %000.5fs  |
+-----------+----------+-----------+""" % (
    sys.version,
    memoize_read_time,
    lru_cache_read_time,
    memoize_write_time,
    lru_cache_write_time
))
