# -*- coding: utf-8 -*-
""" Este módulo é responsável por toda a parte cadastral básica do controle 
interno.

Durante o uso do sistema, este módulo pode ser considerado o "ponto de entrada"
pois é aqui que são cadastradas todas as informações essenciais usadas pelos
outros módulos/aplicações.

.. moduleauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
.. moduleauthor:: Fernando José Loureiro <nando_al@msn.com>
"""

from __future__ import unicode_literals

import os
import shutil

from django.conf import settings
from django.db import models
from django.db.models import signals
from django.db.models.loading import get_models

from ci.fields import CharTitleField
from ci.fields import CharLowerCaseField

from ..models import Base
from ..utils.utils import mkdir
from ..utils.utils import execute_sql
from ..utils.utils import to_ascii

from .consts import VERBOSE_ANO
from .consts import VERBOSE_ANO_PLURAL
from .consts import VERBOSE_EMAIL
from .consts import VERBOSE_EMAIL_PLURAL
from .consts import VERBOSE_MES
from .consts import VERBOSE_MES_PLURAL
from .consts import VERBOSE_MES_CODIGO
from .consts import VERBOSE_MES_ESPECIAL
from .consts import VERBOSE_OPERADORA_TELEFONIA
from .consts import VERBOSE_OPERADORA_TELEFONIA_PLURAL
from .consts import VERBOSE_TELEFONE
from .consts import VERBOSE_TELEFONE_PLURAL
from ci.download.models import DownloadSemCategoria
from ci.cadastro.consts import VERBOSE_SISTEMA, VERBOSE_SISTEMA_PLURAL
from ci.cadastro.tipo.models import TipoHierarquiaBase


class Ano(Base):
    """ Cadastro de anos.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    """
    nome = models.IntegerField(unique = True,
                               verbose_name = 'ConfigurationField.objects.get(config_name="verbose_name")',)

#    def __init__(self, *args, **kwargs):
#        super(Ano, self).__init__(*args, **kwargs)
#        for field in self._meta.fields:
#            print(field.verbose_name)
#            field.verbose_name = "Teste"

    class Meta:
        ordering = ['nome',]
        verbose_name = VERBOSE_ANO
        verbose_name_plural = VERBOSE_ANO_PLURAL
        db_table = "cadastro_ano"

    def __unicode__(self):
        return unicode(self.nome)


class Mes(Base):
    """ Cadastro de meses.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    """
    nome = CharTitleField(max_length = 32,
                          unique = True,
                          verbose_name = VERBOSE_MES,)
    codigo = models.IntegerField(unique = True,
                                 verbose_name = VERBOSE_MES_CODIGO,)
    mes_especial = models.BooleanField(default = False,
                                       verbose_name = VERBOSE_MES_ESPECIAL,)

    class Meta:
        ordering = ['codigo']
        verbose_name = VERBOSE_MES
        verbose_name_plural = VERBOSE_MES_PLURAL
        db_table = "cadastro_mes"

    def __unicode__(self):
        return self.nome


class OperadoraTelefonia(Base):
    """ Cadastro de operadoras telefônicas.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    """
    nome = CharTitleField(max_length = 64,
                          unique = True,
                          verbose_name = VERBOSE_OPERADORA_TELEFONIA,)

    class Meta:
        ordering = ['nome',]
        verbose_name = VERBOSE_OPERADORA_TELEFONIA
        verbose_name_plural = VERBOSE_OPERADORA_TELEFONIA_PLURAL
        db_table = "cadastro_operadora_telefonia"

    def __unicode__(self):
        return self.nome


class Email(Base):
    """ Cadastro de e-mails.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    """
    endereco = CharLowerCaseField(db_index = True,
                                  max_length = 64,
                                  verbose_name = VERBOSE_EMAIL,)

    class Meta:
        abstract = True
        verbose_name = VERBOSE_EMAIL
        verbose_name_plural = VERBOSE_EMAIL_PLURAL

    def __unicode__(self):
        return self.endereco


class Telefone(Base):
    """ Cadastro de números telefônicos.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    """
    numero = models.CharField(db_index = True,
                              max_length = 9,
                              verbose_name = VERBOSE_TELEFONE,)

    class Meta:
        abstract = True
        verbose_name = VERBOSE_TELEFONE
        verbose_name_plural = VERBOSE_TELEFONE_PLURAL


class SistemaCategoria(TipoHierarquiaBase):
    class Meta:
        db_table = "cadastro_sistema_categoria"


class Sistema(DownloadSemCategoria):
    """ Cadastro de sistemas.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>

    Esta classe representa um software que a Tavares & Souza pode alugar para
    um cliente, como por exemplo um sistema de controle tributário.

    Qualquer local do controle interno que precise fazer refência a um 
    sistema(e não a qualquer arquivo disponível para download) deve criar uma 
    referência para esta classe.

    Poderia ter sido usada a classe DownloadSemCategoria para esta função, porém foi 
    decidido criar esta classe especializada para aumentar a flexibilidade do 
    sistema caso seja necessário incluir novas informações para um sistema.
    
    """
    categoria = models.ForeignKey(SistemaCategoria)

    class Meta:
        verbose_name = VERBOSE_SISTEMA
        verbose_name_plural = VERBOSE_SISTEMA_PLURAL
        db_table = "cadastro_sistema"


def mes_pre_save(signal, instance, sender, **kwargs):
    """ Mantém a consistência dos arquivos no sistema de arquivos. 

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Fernando José Loureiro <nando_al@msn.com>

    Este método garante que caso o nome ou o código de um mês seja alteração 
    os arquivos vínculados a ele sejam movidos para uma nova localização que 
    seja condizente com o nome/código do mês. Isso é feito para manter a 
    consistência do sistema de arquivos, já que o path dos arquivos é montado 
    com base no nome do mês e do código(além de outros dados).
    """
    if instance.id:
        protocolos_arquivo = instance.protocolorelatorio_set.all()
        old_full_path = None

        if len(unicode(instance.codigo)) == 1:
            mes = '0' + unicode(instance.codigo)
        else:
            mes = unicode(instance.codigo)

        mes = mes + '_' + instance.nome

        for protocolo_relatorio in protocolos_arquivo:
            for protocolo_relatorio_arquivo in protocolo_relatorio.protocolorelatorioarquivo_set.all():
                old_full_path = protocolo_relatorio_arquivo.protocolo_relatorio_arquivo.path

                old_relative_path = protocolo_relatorio_arquivo.protocolo_relatorio_arquivo.name

                novo_dst = os.path.join(old_relative_path.rsplit('/', 2)[0],
                                        mes,
                                        old_relative_path.rsplit('/', 1)[-1]
                                        )

                sql = 'update ' + protocolo_relatorio_arquivo._meta.db_table +\
                ' set ' + protocolo_relatorio_arquivo.protocolo_relatorio_arquivo.field.column + '=\'' +\
                novo_dst + '\' where id=' + unicode(protocolo_relatorio_arquivo.id)

                execute_sql(sql.lower())

            src = old_full_path.rsplit('/', 1)[0]
            dst = os.path.join(old_full_path.rsplit('/', 2)[0], mes)
            dst = to_ascii(dst)

            shutil.move(src.lower(), dst.lower())


def generate_custom_sql(**kwargs):
    """ Cria um arquivo sql para criar esquemas no postgres.

    .. sectionauthor:: Elton Pereira de Lima <eltonplima@gmail.com>
    .. codeauthor:: Fernando José Loureiro <nando_al@msn.com>

    Este método é invocado após a execução do syncdb. Ele gera um arquivo com o 
    nome que foi setado em settings.FIXTURE_FILE_NAME, dentro do diretório: 
    project_name/app_name/sql/nome_do_arquivo.
    """
    database_engine = settings.DATABASES['default']['ENGINE']

    if database_engine.find('postgresql') != 0:
        pass

    app = kwargs['app']
    package = app.__package__

    if package is not None:
        PROJECT_APPS = list(settings.PROJECT_APPS)
        EXTERNAL_APPS = list(settings.EXTERNAL_APPS)
        try:
            app_name = package.split('.')[1]
            package = package.split('.')[-1]
            FIXTURE_PATH = os.path.join(settings.PROJECT_ROOT_PATH, app_name, 'sql')
            FIXTURE_FILE = os.path.join(FIXTURE_PATH, settings.FIXTURE_FILE_NAME)
            PRE_FIXTURE_FILE = os.path.join(FIXTURE_PATH, "pre_fixtures.sql")
            POS_FIXTURE_FILE = os.path.join(FIXTURE_PATH, "pos_fixtures.sql")
            PROJECT_APPS = [p.split('.')[-1] for p in PROJECT_APPS]
            query_external_apps_move = None       

            if EXTERNAL_APPS.index(package) >= 0:
                query_external_apps_move = "select func_movetablestoschema('%s');" % package
        except ValueError:
            pass
        except IndexError:
            pass

        try:
            if PROJECT_APPS.index(package) >= 0:
                                
                master_apps = []
                for p in list(settings.PROJECT_APPS):                    
                    #coloca apenas as apps que tenham models
                    modelos = get_models()
                    for modelo in modelos:
                        if modelo.__module__.startswith(p):
                            p = p.split('.')[1]                                     
                            if p not in master_apps:              
                                    master_apps.append(p)                    
                
                apps = ', '.join(master_apps)
                apps_audit = ''
                
                for app in master_apps:                    
                    apps_audit += app + '_audit, '

                apps_audit = apps_audit.rsplit(',', 1)[0]
                query_search_path = 'alter user %s set search_path to public, django, %s, %s;' % (settings.DATABASES['default']['USER'], apps_audit, apps)
                
                mkdir(FIXTURE_PATH)
                query = "select func_movetablestoschema('%s');\n" % app_name
                query += 'select func_movetablestoschema(\'audit\' ,\'%s\', \'audit\');' % (app_name)
                fixture_file = open(FIXTURE_FILE,"w")

                # Envia o conteúdo do arquivo pre_fixtures.sql para o inicio
                if os.path.isfile(PRE_FIXTURE_FILE):
                    pre_fixture_file = open(PRE_FIXTURE_FILE,"r")

                    for line in pre_fixture_file.readlines():
                        fixture_file.write(line)

                    pre_fixture_file.close()

                fixture_file.write(query+'\n')

                # Envia o conteúdo do arquivo pos_fixtures.sql para o final
                if os.path.isfile(POS_FIXTURE_FILE):
                    pos_fixture_file = open(POS_FIXTURE_FILE,"r")

                    for line in pos_fixture_file.readlines():
                        fixture_file.write(line)

                    pos_fixture_file.close()

                if query_external_apps_move is not None:
                    fixture_file.write(query_external_apps_move + '\n')

                fixture_file.write('select func_movetablestoschema(\'django\');\n')
                fixture_file.write('select func_movetablestoschema(\'auth\' ,\'django\');\n')
                fixture_file.write('select func_delete_foreignkey_audit();\n')
                fixture_file.write(query_search_path + '\n')
                fixture_file.close()
        except ValueError:
            pass

signals.post_syncdb.connect(generate_custom_sql)
#signals.pre_save.connect(mes_pre_save, sender=Mes, dispatch_uid='mes_pre_save')
