# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.db import models

from ci.consts import VERBOSE_GRUPO_PLURAL
from ci.fields import CharTitleField
from ci.fields import CharLowerCaseField
from ci.models import Base

from ..consts import VERBOSE_OPERADORA_TELEFONIA
from ..endereco.consts import VERBOSE_LOGRADOURO
from ..endereco.models import Logradouro
from ..models import Email
from ..models import Telefone
from ..tipo.consts import VERBOSE_TIPO_ENDERECO
from ..tipo.consts import VERBOSE_TIPO_EMAIL
from ..tipo.consts import VERBOSE_TIPO_PAGINAWEB
from ..tipo.consts import VERBOSE_TIPO_TELEFONE
from ..tipo.models import TipoEmail
from ..tipo.models import TipoEndereco
from ..tipo.models import TipoPaginaWeb
from ..tipo.models import TipoTelefone
from ..models import OperadoraTelefonia

from .consts import VERBOSE_CARGO
from .consts import VERBOSE_CARGO_PLURAL
from .consts import VERBOSE_EMAIL_PESSOA
from .consts import VERBOSE_EMAIL_PESSOA_PLURAL
from .consts import VERBOSE_ENDERECO_PESSOA
from .consts import VERBOSE_ENDERECO_PESSOA_PLURAL
from .consts import VERBOSE_ENDERECO_ELETRONICO
from .consts import VERBOSE_NOME
from .consts import VERBOSE_PAGINA_WEB_PESSOA
from .consts import VERBOSE_PAGINA_WEB_PESSOA_PLURAL
from .consts import VERBOSE_PESSOA_BASE
from .consts import VERBOSE_PESSOA_BASE_PLURAL
from .consts import VERBOSE_RESPONSAVEL_FINANCEIRO
from .consts import VERBOSE_RESPONSAVEL_FINANCEIRO_PLURAL
from .consts import VERBOSE_TELEFONE_PESSOA
from .consts import VERBOSE_TELEFONE_PESSOA_PLURAL
from .consts import VERBOSE_USUARIO
from .consts import VERBOSE_USUARIO_PLURAL
from .consts import VERBOSE_USUARIO_TERMINAL_SERVICE
from .consts import VERBOSE_USUARIO_TERMINAL_SERVICE_PLURAL
from .consts import VERBOSE_USUARIO_TERMINAL_SERVICE_ATIVO
from .consts import VERBOSE_PESSOA_BASE_TIPO_PESSOA
from .consts import VERBOSE_NUMERO
from .consts import VERBOSE_COMPLEMENTO
from .consts import VERBOSE_EMAIL_PESSOA_PADRAO
from .consts import VERBOSE_ENDERECO_PESSOA_PADRAO
from .consts import VERBOSE_TELEFONE_PESSOA_PADRAO
from django.contrib.auth.models import AbstractBaseUser


class PessoaBase(Base):
    nome = CharTitleField(db_index = True,
                          max_length = 255,
                          verbose_name = VERBOSE_NOME,)
    observacao = models.TextField(null=True, blank=True)

    class Meta:
        db_table = 'cadastro_pessoa'
        ordering = ['nome',]
        verbose_name = VERBOSE_PESSOA_BASE
        verbose_name_plural = VERBOSE_PESSOA_BASE_PLURAL 
        
    def get_tipo_pessoa(self):
        try:
            retorno = self.pessoafisica.cpf
        except:
            try:                
                retorno = self.pessoajuridica.cnpj
            except:
                retorno = ''
        return '%s (%s)' % (self.nome,
                            retorno,) 
    get_tipo_pessoa.short_description = VERBOSE_PESSOA_BASE_TIPO_PESSOA
    
    def get_endereco_padrao(self):
        """ Retorna o endereço padrão da pessoa.
        """
        endereco = None

        try:
            endereco = self.enderecopessoa_set.get(padrao=True)
        except: 
            pass

        return endereco
    get_endereco_padrao.short_description = VERBOSE_ENDERECO_PESSOA 

    def get_municipio(self):
        return self.get_endereco_padrao().logradouro.bairro.municipio

    def get_estado(self):
        return self.get_municipio().estado

    def get_telefone_padrao(self):
        """ Retorna o telefone padrão da pessoa.
        """
        telefone = "Nenhum"

        try:
            telefone = self.telefonepessoa_set.get(padrao=True)
        except: 
            pass

        return telefone
    get_telefone_padrao.short_description = VERBOSE_TELEFONE_PESSOA
    
    def get_email_padrao(self):
        """ Retorna o email padrão da pessoa.
        """
        email = "Nenhum"

        try:
            email = self.emailpessoa_set.get(padrao=True)
        except: 
            pass

        return email
    get_email_padrao.short_description = VERBOSE_EMAIL_PESSOA
    
    def get_paginaweb_padrao(self):
        """ Retorna a página web padrão da pessoa.
        """
        paginaweb = "Nenhum"

        try:
            paginaweb = self.paginawebpessoa_set.get(padrao=True)
        except: 
            pass

        return paginaweb
    get_paginaweb_padrao.short_description = VERBOSE_PAGINA_WEB_PESSOA
    
    def __unicode__(self):                
        return self.get_tipo_pessoa()


class Cargo(Base):
    nome = CharTitleField(max_length = 128,
                          unique = True,
                          verbose_name = VERBOSE_CARGO,)

    class Meta:
        db_table = 'cadastro_pessoa_cargo'
        ordering = ['nome',]
        verbose_name = VERBOSE_CARGO
        verbose_name_plural = VERBOSE_CARGO_PLURAL

    def __unicode__(self):
        return self.nome  


class EmailPessoa(Email):
    tipo_email = models.ForeignKey(TipoEmail,
                                   verbose_name = VERBOSE_TIPO_EMAIL,)
    padrao = models.BooleanField(unique = True,
                                 verbose_name = VERBOSE_EMAIL_PESSOA_PADRAO,)
    pessoa_base = models.ForeignKey(PessoaBase,
                                    verbose_name = VERBOSE_PESSOA_BASE,)

    class Meta:
        db_table = 'cadastro_pessoa_email'
        ordering = ['tipo_email',]
        verbose_name = VERBOSE_EMAIL_PESSOA
        verbose_name_plural = VERBOSE_EMAIL_PESSOA_PLURAL

    def __unicode__(self):
        return self.endereco


class TelefonePessoa(Telefone):
    tipo_telefone = models.ForeignKey(TipoTelefone,
                                      verbose_name = VERBOSE_TIPO_TELEFONE,)
    operadora_telefonia = models.ForeignKey(OperadoraTelefonia,
                                            verbose_name = VERBOSE_OPERADORA_TELEFONIA,)
    pessoa_base = models.ForeignKey(PessoaBase,
                                    verbose_name = VERBOSE_PESSOA_BASE,)
    padrao = models.BooleanField(unique = True,
                                 verbose_name = VERBOSE_TELEFONE_PESSOA_PADRAO,)

    class Meta:
        db_table = 'cadastro_pessoa_telefone'
        ordering = ['tipo_telefone',]
        verbose_name = VERBOSE_TELEFONE_PESSOA
        verbose_name_plural = VERBOSE_TELEFONE_PESSOA_PLURAL

    def __unicode__(self):
        return self.numero


class EnderecoPessoa(Base):
    tipo_endereco = models.ForeignKey(TipoEndereco,
                                      verbose_name = VERBOSE_TIPO_ENDERECO,)
    pessoa_base = models.ForeignKey(PessoaBase,
                                    verbose_name = VERBOSE_PESSOA_BASE,)
    numero = models.CharField(default = "S/N",
                              max_length = 64, 
                              verbose_name = VERBOSE_NUMERO,)
    complemento = models.TextField(blank = True, 
                                   null = True,
                                   verbose_name = VERBOSE_COMPLEMENTO,)
    logradouro = models.ForeignKey(Logradouro,
                                   verbose_name = VERBOSE_LOGRADOURO)
    padrao = models.BooleanField(verbose_name = VERBOSE_ENDERECO_PESSOA_PADRAO)

    class Meta:
        db_table = 'cadastro_pessoa_endereco'
        ordering = ['logradouro',]
        verbose_name = VERBOSE_ENDERECO_PESSOA
        verbose_name_plural = VERBOSE_ENDERECO_PESSOA_PLURAL

    def __unicode__(self):
        return self.logradouro.nome


class PaginaWEBPessoa(Base):
    tipo_pagina_web = models.ForeignKey(TipoPaginaWeb,
                                        verbose_name = VERBOSE_TIPO_PAGINAWEB,)
    pessoa_base = models.ForeignKey(PessoaBase,
                                    verbose_name = VERBOSE_PESSOA_BASE,)
    endereco_eletronico = CharLowerCaseField(db_index = True,
                                             max_length = 255,
                                             verbose_name = VERBOSE_ENDERECO_ELETRONICO,)

    class Meta:
        db_table = 'cadastro_pessoa_paginaweb'
        ordering = ['endereco_eletronico',]
        verbose_name = VERBOSE_PAGINA_WEB_PESSOA
        verbose_name_plural = VERBOSE_PAGINA_WEB_PESSOA_PLURAL

    def __unicode__(self):
        return self.endereco_eletronico


class Usuario(AbstractBaseUser):
    """ Víncula uma pessoa como sendo do tipo usuário.
    """
    pessoa_base = models.ForeignKey(PessoaBase,
                                    unique = True,
                                    verbose_name = VERBOSE_PESSOA_BASE,)

    class Meta:
        db_table = 'cadastro_pessoa_usuario'
        ordering = ['pessoa_base',]
        verbose_name = VERBOSE_USUARIO
        verbose_name_plural = VERBOSE_USUARIO_PLURAL

    def __unicode__(self):
        return self.pessoa_base.get_tipo_pessoa()

    def get_grupos(self):
        return ', '.join([grupo.name for grupo in self.groups.all()])
    get_grupos.short_description=VERBOSE_GRUPO_PLURAL


class UsuarioTerminalService(Base):
    """ Víncula uma pessoa como usuário do terminal service.
    
    Além de víncular uma pessoa como usuário do serviço de terminal service,
    ainda pode ser controlado se a mesma está ou não ativa no serviço.
    """
    pessoa_base = models.ForeignKey(PessoaBase,                                    
                                    unique = True,
                                    verbose_name = VERBOSE_USUARIO_TERMINAL_SERVICE,)
    ativo = models.BooleanField(default = True,
                                verbose_name = VERBOSE_USUARIO_TERMINAL_SERVICE_ATIVO,)

    class Meta:
        db_table = 'cadastro_pessoa_usuarioterminalservice'
        ordering = ['pessoa_base',]
        verbose_name = VERBOSE_USUARIO_TERMINAL_SERVICE
        verbose_name_plural = VERBOSE_USUARIO_TERMINAL_SERVICE_PLURAL

    def __unicode__(self):
        return self.pessoa_base.get_tipo_pessoa()


class ResponsavelFinanceiro(Base):
    """ Pessoa responsável por pagar as licenças dos sistemas.
    """
    pessoa_base = models.ForeignKey(PessoaBase, 
                                    unique = True,
                                    verbose_name = VERBOSE_RESPONSAVEL_FINANCEIRO,)

    class Meta:
        db_table = 'cadastro_pessoa_responsavelfinanceiro'
        ordering = ['pessoa_base',]
        verbose_name = VERBOSE_RESPONSAVEL_FINANCEIRO
        verbose_name_plural = VERBOSE_RESPONSAVEL_FINANCEIRO_PLURAL

    def __unicode__(self):
        return self.pessoa_base.get_tipo_pessoa()
