from django.db import models
from django.utils.translation import ugettext_lazy as _

class ServiceProvider(models.Model):
    title = models.CharField(_("Title"), max_length=150, unique=True)

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)
        verbose_name = _("Service provider")
        verbose_name_plural = _("Service providers")


class Service(models.Model):
    provider = models.ForeignKey("ServiceProvider",
                                 verbose_name=_("Service provider"),
                                 related_name="services")
    title = models.CharField(_("Title"), max_length=150)
    provider_discount = models.DecimalField(_("Provider discount"),
                                            max_digits=4,
                                            decimal_places=2, default=0)

class EntertainmentType(models.Model):
    title = models.CharField(max_length=150, unique=True)

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)


class Entertainment(Service):
    types = models.ManyToManyField(EntertainmentType, verbose_name=_("Types"),
                                   related_name="entertainments")

    description = models.TextField(_("Description"), blank=True)
    season = models.TextField(_("Season"), blank=True)
    clothing = models.TextField(_("Clothing"), blank=True)
    weather_conditions = models.TextField(_("Weather conditions"), blank=True)
    duration = models.TextField(_("Duration"), blank=True)
    location = models.TextField(_("Location"), blank=True)

    min_people = models.PositiveIntegerField(_("Min. number of people"),
                                             blank=True,
                                             null=True, default=None)
    max_people = models.PositiveIntegerField(_("Max. number of people"),
                                             blank=True,
                                             null=True, default=None)
    price_for_person = models.DecimalField(_("Price for person"), blank=True,
                                           null=True, default=None,
                                           decimal_places=2, max_digits=7)
    price_for_service = models.DecimalField(_("Price for service"), blank=True,
                                            null=True, default=None,
                                            decimal_places=2, max_digits=7)
    website = models.URLField(_("Website"), blank=True)

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)


class EntertainmentPhoto(models.Model):
    entertainment = models.ForeignKey(Entertainment, related_name="photos")
    description = models.TextField(_("Description"), max_length=1000,
                                   blank=True)
    photo = models.ImageField(_("Photo"), upload_to="services/photos/entertainment")
    is_main = models.BooleanField(verbose_name=_("Main"), default=False)

    def get_photo_tag(self):
        return '<img src="%s" title="%s" alt="%s" width="100" height="100"/>' % (
            self.photo.url, self.description, self.description)

    class Meta:
        ordering = ('-is_main',)
        verbose_name = _("Photo")
        verbose_name_plural = _("Photos")
