#### Example use case: object permission check wrapper for view classes

# This can be wrapped at a class level to provide a simple @permissions(perms_list)
# decorator for class views that can be used to check request user for 
# permissions on the objects returned by the view 

def func_permissions(*perms):
    """ Class function decorator to check user for object permissions
        User must have all the permissions listed to get access. 
    """
    def decorator(view_func):
        @wraps(view_func, assigned=available_attrs(view_func))
        def _inner(request=None, *args, **kwargs):
            """ when decorating a class view_func is get_context_data
                NB: requires dispatch decorator to set request.user in kwargs
            """
            obj_list = kwargs.get('object_list', [])
            context = view_func(**kwargs)
            if not obj_list:
                obj = context.get('subobject', 
                                  context.get('object', None))
                if obj:
                    obj_list = [obj, ]
            user = kwargs.get('user', context.get('user', None))
            if not user:
                raise Denied("""No user has been passed in kwargs or context 
                                      to test %s permissions""" % str(perms))
            if not obj_list:
                raise Denied("""There is no object supplied in the request 
                                      to test %s permissions""" % str(perms))
            for codename in perms:
                for obj in obj_list:
                    if not has_permission(obj, user, codename):
                        raise Denied("""User '%s' doesn't have permission 
                                          '%s' for object '%s' (%s)""" 
                                   % (user, codename, obj, obj.__class__.__name__))
            return view_func(*args, **kwargs)
        return _inner
    return decorator

############## Monkey patch view classes to pass kwargs for decorators #####

def processformview_get(self, request, *args, **kwargs):
    form_class = self.get_form_class()
    form = self.get_form(form_class)
    kwargs['form'] = form
    context = self.get_context_data(**kwargs)
    return self.render_to_response(context)

def baselistview_get(self, request, *args, **kwargs):
    self.object_list = self.get_queryset()
    allow_empty = self.get_allow_empty()
    if not allow_empty and len(self.object_list) == 0:
        raise Http404(_(u"Empty list and '%(class_name)s.allow_empty' is False.")
                      % {'class_name': self.__class__.__name__})
    kwargs['object_list'] = self.object_list
    context = self.get_context_data(**kwargs)
    return self.render_to_response(context)

def basedetailview_get(self, request, *args, **kwargs):
    self.object = self.get_object()
    kwargs['object'] = self.object    
    context = self.get_context_data(**kwargs)
    return self.render_to_response(context)

def basedatelistview_get(self, request, *args, **kwargs):
    self.date_list, self.object_list, extra_context = self.get_dated_items()
    kwargs['object_list'] = self.object_list
    kwargs['date_list'] = self.date_list
    context = self.get_context_data(**kwargs)
    context.update(extra_context)
    return self.render_to_response(context)

from django.views.generic.edit import ProcessFormView
ProcessFormView.get = processformview_get
from django.views.generic.list import BaseListView
BaseListView.get = baselistview_get
from django.views.generic.detail import BaseDetailView
BaseDetailView.get = basedetailview_get
from django.views.generic.dates import BaseDateListView
BaseDateListView.get = basedatelistview_get


