# Copyright (c) 2007, Robert Coup <robert.coup@onetrackmind.co.nz>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without modification,
# are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Distance nor the names of its contributors may be used
#      to endorse or promote products derived from this software without
#      specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
# ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""
Distance and Area objects to allow for sensible and convienient calculation
and conversions.

Authors: Robert Coup, Justin Bronn

Inspired by GeoPy (http://exogen.case.edu/projects/geopy/)
and Geoff Biggs' PhD work on dimensioned units for robotics.
"""
__all__ = ['A', 'Area', 'D', 'Distance', 'V', 'Volume', 'W', 'Weight']
from decimal import Decimal

def is_number(obj):
    return isinstance(obj, (int, float, long, Decimal))
    
def pretty_name(obj):
    if obj.__class__ == type:
        return obj.__name__
    return obj.__class__.__name__

class MeasureBase(object):
    
    STANDARD_UNIT  = None
    ALIAS  = {}
    UNITS  = {}
    LALIAS = {}
    
    def __init__(self, default_unit=None, **kwargs):
        # The base unit is in meters.
        value, self._default_unit = self.default_units(kwargs)
        setattr(self, self.STANDARD_UNIT, value)
        if default_unit and isinstance(default_unit, str):
            self._default_unit = default_unit
            
    def _get_standard(self):
        return getattr(self, self.STANDARD_UNIT)
        
    def _set_standard(self, value):
        setattr(self, self.STANDARD_UNIT, value)
        
    standard = property(_get_standard, _set_standard)
            
    def __getattr__(self, name):
        if name in self.UNITS:
            return self.standard / self.UNITS[name]
        else:
            raise AttributeError('Unknown unit type: %s' % name)

    def __repr__(self):
        return '%s(%s=%s)' % (pretty_name(self), self._default_unit, getattr(self, self._default_unit))

    def __str__(self):
        return '%s %s' % (getattr(self, self._default_unit), self._default_unit)

    def __cmp__(self, other):
        if isinstance(other, self.__class__):
            return cmp(self.m, other.m)
        else:
            return NotImplemented

    def __add__(self, other):
        if isinstance(other, self.__class__):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard + other.standard)})
        else:
            raise TypeError('%(class)s must be added with %(class)s' % {"class":pretty_name(self)})

    def __iadd__(self, other):
        if isinstance(other, self.__class__):
            self.standard += other.standard
            return self
        else:
            raise TypeError('%(class)s must be added with %(class)s' % {"class":pretty_name(self)})

    def __sub__(self, other):
        if isinstance(other, self.__class__):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard - other.standard)})
        else:
            raise TypeError('%(class)s must be subtracted from %(class)s' % {"class":pretty_name(self)})

    def __isub__(self, other):
        if isinstance(other, self.__class__):
            self.standard -= other.standard
            return self
        else:
            raise TypeError('%(class)s must be subtracted from %(class)s' % {"class":pretty_name(self)})

    def __mul__(self, other):
        if is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard * other)})
        else:
            raise TypeError('%(class)s must be multiplied with number' % {"class":pretty_name(self)})

    def __imul__(self, other):
        if is_number(other):
            self.standard *= float(other)
            return self
        else:
            raise TypeError('%(class)s must be multiplied with number' % {"class":pretty_name(self)})

    def __rmul__(self, other):
        return self * other

    def __div__(self, other):
        if isinstance(other, self.__class__):
            return self.standard / other.standard      
        if is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard / other)})
        else:
            raise TypeError('%(class)s must be divided with number or %(class)s' % {"class":pretty_name(self)})

    def __idiv__(self, other):
        if is_number(other):
            self.standard /= float(other)
            return self
        else:
            raise TypeError('%(class)s must be divided with number' % {"class":pretty_name(self)})

    def __nonzero__(self):
        return bool(self.default)            
    
    def default_units(self, kwargs):
        """
        Return the unit value and the default units specified
        from the given keyword arguments dictionary.
        """
        val = 0.0
        for unit, value in kwargs.iteritems():
            if not isinstance(value, float): value = float(value)
            if unit in self.UNITS:
                val += self.UNITS[unit] * value
                default_unit = unit
            elif unit in self.ALIAS:
                u = self.ALIAS[unit]
                val += self.UNITS[u] * value
                default_unit = u
            else:
                lower = unit.lower()
                if lower in self.UNITS:
                    val += self.UNITS[lower] * value
                    default_unit = lower
                elif lower in self.LALIAS:
                    u = self.LALIAS[lower]
                    val += self.UNITS[u] * value
                    default_unit = u
                else:
                    raise AttributeError('Unknown unit type: %s' % unit)
        return val, default_unit

    @classmethod
    def unit_attname(cls, unit_str):
        """
        Retrieves the unit attribute name for the given unit string.
        For example, if the given unit string is 'metre', 'm' would be returned.
        An exception is raised if an attribute cannot be found.
        """
        lower = unit_str.lower()
        if unit_str in cls.UNITS:
            return unit_str
        elif lower in cls.UNITS:
            return lower
        elif lower in cls.LALIAS:
            return cls.LALIAS[lower]
        else:
            raise Exception('Could not find a unit keyword associated with "%s"' % unit_str)

class Distance(MeasureBase):
    STANDARD_UNIT = "m"
    UNITS = {
        'chain' : 20.1168,
        'chain_benoit' : 20.116782,
        'chain_sears' : 20.1167645,
        'british_chain_benoit' : 20.1167824944,
        'british_chain_sears' : 20.1167651216,
        'british_chain_sears_truncated' : 20.116756,
        'cm' : 0.01,
        'british_ft' : 0.304799471539,
        'british_yd' : 0.914398414616,
        'clarke_ft' : 0.3047972654,
        'clarke_link' : 0.201166195164,
        'fathom' :  1.8288,
        'ft': 0.3048,
        'german_m' : 1.0000135965,
        'gold_coast_ft' : 0.304799710181508,
        'indian_yd' : 0.914398530744,
        'inch' : 0.0254,
        'km': 1000.0,
        'link' : 0.201168,
        'link_benoit' : 0.20116782,
        'link_sears' : 0.20116765,
        'm': 1.0,
        'mi': 1609.344,
        'mm' : 0.001,
        'nm': 1852.0,
        'nm_uk' : 1853.184,
        'rod' : 5.0292,
        'sears_yd' : 0.91439841,
        'survey_ft' : 0.304800609601,
        'um' : 0.000001,
        'yd': 0.9144,
        }

    # Unit aliases for `UNIT` terms encountered in Spatial Reference WKT.
    ALIAS = {
        'centimeter' : 'cm',
        'foot' : 'ft',
        'inches' : 'inch',
        'kilometer' : 'km',
        'kilometre' : 'km',
        'meter' : 'm',
        'metre' : 'm',
        'micrometer' : 'um',
        'micrometre' : 'um',
        'millimeter' : 'mm',
        'millimetre' : 'mm',
        'mile' : 'mi',
        'yard' : 'yd',
        'British chain (Benoit 1895 B)' : 'british_chain_benoit',
        'British chain (Sears 1922)' : 'british_chain_sears',
        'British chain (Sears 1922 truncated)' : 'british_chain_sears_truncated',
        'British foot (Sears 1922)' : 'british_ft',
        'British foot' : 'british_ft',
        'British yard (Sears 1922)' : 'british_yd',
        'British yard' : 'british_yd',
        "Clarke's Foot" : 'clarke_ft',
        "Clarke's link" : 'clarke_link',
        'Chain (Benoit)' : 'chain_benoit',
        'Chain (Sears)' : 'chain_sears',
        'Foot (International)' : 'ft',
        'German legal metre' : 'german_m',
        'Gold Coast foot' : 'gold_coast_ft',
        'Indian yard' : 'indian_yd',
        'Link (Benoit)': 'link_benoit',
        'Link (Sears)': 'link_sears',
        'Nautical Mile' : 'nm',
        'Nautical Mile (UK)' : 'nm_uk',
        'US survey foot' : 'survey_ft',
        'U.S. Foot' : 'survey_ft',
        'Yard (Indian)' : 'indian_yd',
        'Yard (Sears)' : 'sears_yd'
        }
    LALIAS = dict([(k.lower(), v) for k, v in ALIAS.items()])

    def __mul__(self, other):
        if isinstance(other, Area):
            return Volume(default_unit=VOLUME_PREFIX + self._default_unit, 
                **{VOLUME_PREFIX + self.STANDARD_UNIT: (self.standard * other.standard)})        
        elif isinstance(other, self.__class__):
            return Area(default_unit=AREA_PREFIX + self._default_unit, 
                **{AREA_PREFIX + self.STANDARD_UNIT: (self.standard * other.standard)})
        elif is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard * other)})
        else:
            raise TypeError('%(distance)s must be multiplied with number, %(distance)s or %(area)s' % {
                "distance" : pretty_name(self.__class__),
                "area"     : pretty_name(Area),
                })
            
AREA_PREFIX = "sq_"

class Area(MeasureBase):
    STANDARD_UNIT = AREA_PREFIX + Distance.STANDARD_UNIT
    # Getting the square units values and the alias dictionary.
    UNITS = dict([(AREA_PREFIX + k, v ** 2) for k, v in Distance.UNITS.items()])
    ALIAS = dict([(k, AREA_PREFIX + v) for k, v in Distance.ALIAS.items()])
    LALIAS = dict([(k.lower(), v) for k, v in ALIAS.items()])

    def __div__(self, other):
        if isinstance(other, self.__class__):
            return self.standard / other.standard          
        if isinstance(other, Distance):
            return Distance(default_unit=self._default_unit[len(AREA_PREFIX):], 
                **{Distance.STANDARD_UNIT: (self.standard / other.standard)})
        elif is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard / other)})
        else:
            raise TypeError('%(area)s must be divided with number, %(distance)s or %(area)s' % {
                "distance" : pretty_name(Distance),
                "area"     : pretty_name(self.__class__),
                }) 
            
    def __mul__(self, other):
        if isinstance(other, Distance):
            return Volume(default_unit=VOLUME_PREFIX+self._default_unit[len(AREA_PREFIX):], 
                **{Volume.STANDARD_UNIT: (self.standard * other.standard)})        
        elif is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard * other)})
        else:
            raise TypeError('%(area)s must be multiplied with number or %(distance)s' % {
                "distance" : pretty_name(Distance),
                "area"     : pretty_name(self.__class__),
                }) 
            
VOLUME_PREFIX = "vol_"

class Volume(MeasureBase):
    STANDARD_UNIT = VOLUME_PREFIX + Distance.STANDARD_UNIT
    # Getting the cube units values and the alias dictionary.
    UNITS = dict([(VOLUME_PREFIX + k, v ** 3) for k, v in Distance.UNITS.items()])
    ALIAS = dict([(k, VOLUME_PREFIX + v) for k, v in Distance.ALIAS.items()])
    LALIAS = dict([(k.lower(), v) for k, v in ALIAS.items()])    
    
    def __div__(self, other):
        if isinstance(other, self.__class__):
            return self.standard / other.standard          
        if isinstance(other, Area):
            return Distance(default_unit=self._default_unit[len(VOLUME_PREFIX):], 
                **{Distance.STANDARD_UNIT: (self.standard / other.standard)})        
        if isinstance(other, Distance):
            return Area(default_unit=AREA_PREFIX+self._default_unit[len(VOLUME_PREFIX):], 
                **{Area.STANDARD_UNIT: (self.standard / other.standard)})
        elif is_number(other):
            return self.__class__(default_unit=self._default_unit, 
                **{self.STANDARD_UNIT: (self.standard / other)})
        else:
            raise TypeError('%(volume)s must be divided with number, %(distance)s, %(area)s or %(volume)s' % {
                "distance" : pretty_name(Distance),
                "area"     : pretty_name(Area),
                "volume"   : pretty_name(self.__class__),
                })     
    
class Weight(MeasureBase):
    
    DEFUALT_UNIT = "gr"
    
    UNITS = {
        'mg':  1.0 / 1000,
        'gr':  1.0,
        'kg':  1.0 * 1000,
        'q' :  1.0 * 1000 * 100,
        'ton': 1.0 * 1000 * 1000,
        }

    # Unit aliases for `UNIT` terms encountered in Spatial Reference WKT.
    ALIAS = {
        'Milligram':  'mg',
        'Gram':       'gr',
        'Kilogram':   'kg',
        'Quintal' :   'q',
        'Ton' :       'ton',
        }
    LALIAS = dict([(k.lower(), v) for k, v in ALIAS.items()])

# Shortcuts
D = Distance
A = Area
V = Volume
W = Weight


if __name__ == '__main__':

    from itertools import product
    
    CONSTRUCTORS = {
        float: lambda cls, val: cls(val),
        A:     lambda cls, val: cls(**{cls.STANDARD_UNIT:val}),
        D:     lambda cls, val: cls(**{cls.STANDARD_UNIT:val}),
        V:     lambda cls, val: cls(**{cls.STANDARD_UNIT:val}),
    }
    
    classes = [D, A, V, float]
    
    combs = product(classes, repeat = 2)
    
    a_val = 10
    b_val = 5        
    
    for a_class, b_class in combs:
        
        print "-----------%s-vs-%s-----------" % (a_class.__name__.title(), b_class.__name__.title())
        
        a = CONSTRUCTORS[a_class](a_class, a_val)
        b = CONSTRUCTORS[b_class](b_class, b_val)
        
        for verbose, operator in (("+", "__add__"), ("-", "__sub__"), ("*", "__mul__"), ("/", "__div__")):
            
            desc = "%s %s %s" % (a, verbose, b)
            
            print desc.ljust(25), ">    ", 
            
            try:
                print getattr(a, operator)(b)
            except Exception, e:
                print e    

