from json import dumps
from django import template
from django.core.serializers.json import DjangoJSONEncoder
from django.utils.html import format_html
from django.utils.safestring import mark_safe

register = template.Library()


def escapejson(a):
    """
    This will escape all the HTML/XML special characters with their unicode
    escapes, so it is safe to be output anywhere except for inside a tag
    attribute.
    """
    json_str = dumps(a, cls=DjangoJSONEncoder)

    # Escape all the XML/HTML special characters.
    escapes = ['<', '>', '&']
    for c in escapes:
        json_str = json_str.replace(c, r'\u%04x' % ord(c))

    return mark_safe(json_str)


@register.filter()
def as_json_script(a, id=None):
    """
    This will wrap the escaped json in a script tag with id set accordingly.
    Other javascript can load the data using
    JSON.parse(document.getElementById("<id>").textContent)
    See https://www.owasp.org/index.php/XSS_(Cross_Site_Scripting)_Prevention_Cheat_Sheet#RULE_.233.1_-_HTML_escape_JSON_values_in_an_HTML_context_and_read_the_data_with_JSON.parse
    """

    escaped = escapejson(a)
    if id:
        return format_html(
            '<script id="{}" type="application/json">{}</script>',
            id, escaped
        )
    else:
        return format_html(
            '<script type="application/json">{}</script>',
            escaped
        )
as_json_script.is_safe = True
