from types import MethodType
from django.db import models
from django.db.models.query import QuerySet
from django.db.models.query import EmptyQuerySet

class ManagerQuerySetDecorator(models.Manager):
    """
    Extends the operation methods of the manager on subqueries QuerySet objects.
    
    The "standard manager" you can only like this:
    SomeModel.object.some_method_manager().all().filter(...). ...
    
    But this does not work:
    SomeModel.object.all().some_method_manager()
    
    And this it works with the use of this class / patches:
    SomeModel.object.all().some_method_manager().filter(...).some_method_manager2(). ... 
    SomeModel.object.some_method_manager().some_method_manager2().some_method_manager3(). ... 
    
    Extension of the methods manager on all child objects "QuerySeth".
    
    Example:
    
    class Animals(models.Model):
        type = models.CharField('Type of animal') # dog, cat, elephant
        color = models.CharField('Colour Name')
        
        objects = AnimalsManager()
        
    class AnimalsManager(ManagerQuerySetDecorator):
    
        def get_dogs(self):
            return self.get_query_set().filter(type='dog')
        get_dogs.queryset_method = True
        
        def get_cats(self):
            return self.get_query_set().filter(type='cat')
        get_cats.queryset_method = True
        
        def get_white_animals(self):
            return self.get_query_set().filter(color='white')
        get_white_animals.queryset_method = True
        
        def get_black_animals(self):
            return self.get_query_set().filter(color='black')
        get_black_animals.queryset_method = True
        
        def get_brown_animals(self):
            return self.get_query_set().filter(color='black')

        
    Animals(type='dog', color='black').save()
    Animals(type='dog', color='bown').save()
    Animals(type='dog', color='white').save()
    Animals(type='cat', color='black').save()
    Animals(type='cat', color='bown').save()
    Animals(type='cat', color='white').save()
    
    animals = Animals.objects.get_black_animals().get_dogs()
    # return list black dogs

    animals = Animals.objects.get_white_animals().get_cats()
    # return list white cats
    
    # When ffff equals False, or not defined (as it is in 
    # the original model manager), we can not perform such an operation:
    animals = Animals.objects.get_cats().get_brown_animals()
    # return:
    # AttributeError: 'QuerySet' object has no attribute 'get_brown_animals'
    
    # but:
    animals = Animals.objects.get_brown_animals().get_cats()
    # return brown cats, because get_cat() present in all parent object instances.
    """
    
    def _get_queryset_class(self):
        class ManagerQuerySet(QuerySet):
            def none(self):
                """
                Returns an empty QuerySet.
                """
                klass = type( 
                             "_InstrumentedEmptyQuerySet_%s" % self.__class__.__name__, 
                             (EmptyQuerySet, self.__class__), {}) 
                return self._clone(klass=klass)
        for attrib_name in dir(self):
            attrib = getattr(self,attrib_name)
            if isinstance(attrib, MethodType) and hasattr(attrib, 'queryset_method') and  attrib.queryset_method:
                setattr(ManagerQuerySet, attrib_name, attrib)
        return ManagerQuerySet
    
    def get_query_set(self):
        return self._get_queryset_class()(self.model, using=self._db)
    
    