class _ModelCleanWorkaroundMixin(object):
    """
    Mixin for Model subclasses that wish to raise ValidationErrors
    in Model.clean() that target individual fields. As of Django 2.3
    this is not possible, due to <https://code.djangoproject.com/ticket/16986>.
    """
    
    # Workaround for: https://code.djangoproject.com/ticket/16986
    def clean_fields(self, **kwargs):
        try:
            super(_ModelCleanWorkaroundMixin, self).clean_fields(**kwargs)
        except ValidationError as e1:
            if not hasattr(e1, 'message_dict'):
                e1.message_dict = {NON_FIELD_ERRORS: e1.messages}
        else:
            e1 = None
        
        try:
            # If not for this workaround, this method would be called clean()
            self.clean_self()
        except ValidationError as e2:
            pass
        else:
            e2 = None
        
        if e1 and e2:
            raise ValidationError(dict(e1.message_dict, **e2.message_dict))
        elif e1 and not e2:
            raise e1
        elif not e1 and e2:
            raise e2

def _createLatitudeField():
    return models.DecimalField(max_digits=8, decimal_places=6,
        validators=[
            validators.MinValueValidator(Decimal('-90.000000')),
            validators.MaxValueValidator(Decimal('90.000000')),
        ])

def _createLongitudeField():
    return models.DecimalField(max_digits=9, decimal_places=6,
        validators=[
            validators.MinValueValidator(Decimal('-180.000000')),
            validators.MaxValueValidator(Decimal('180.000000')),
        ])


class Region(_ModelCleanWorkaroundMixin, models.Model):
    south       = _createLatitudeField()
    west        = _createLongitudeField()
    north       = _createLatitudeField()
    east        = _createLongitudeField()
    
    def clean_self(self):
        message_dict = defaultdict(list)
        
        if self.west and self.east and (not self.west <= self.east):
            message_dict['south'].append('Southwest pin must be west of the northeast pin.')
        if self.south and self.north and (not self.south <= self.north):
            message_dict['south'].append('Southwest pin must be south of the northeast pin.')
        
        if len(message_dict):
            raise ValidationError(message_dict)
