Index: django/dispatch/dispatcher.py
===================================================================
--- django/dispatch/dispatcher.py	(revision 16928)
+++ django/dispatch/dispatcher.py	(working copy)
@@ -1,5 +1,6 @@
 import weakref
 import threading
+from collections import defaultdict
 
 from django.dispatch import saferef
 
@@ -13,21 +14,22 @@
 class Signal(object):
     """
     Base class for all signals
-    
+
     Internal attributes:
-    
+
         receivers
             { receriverkey (id) : weakref(receiver) }
     """
-    
+
     def __init__(self, providing_args=None):
         """
         Create a new signal.
-        
+
         providing_args
             A list of the arguments this signal can pass along in a send() call.
         """
-        self.receivers = []
+        self.receivers = defaultdict(list)
+        self.connect_index = 0
         if providing_args is None:
             providing_args = []
         self.providing_args = set(providing_args)
@@ -36,9 +38,9 @@
     def connect(self, receiver, sender=None, weak=True, dispatch_uid=None):
         """
         Connect receiver to sender for signal.
-    
+
         Arguments:
-        
+
             receiver
                 A function or an instance method which is to receive signals.
                 Receivers must be hashable objects.
@@ -46,7 +48,7 @@
                 If weak is True, then receiver must be weak-referencable (more
                 precisely saferef.safeRef() must be able to create a reference
                 to the receiver).
-        
+
                 Receivers must be able to accept keyword arguments.
 
                 If receivers have a dispatch_uid attribute, the receiver will
@@ -62,19 +64,19 @@
                 module will attempt to use weak references to the receiver
                 objects. If this parameter is false, then strong references will
                 be used.
-        
+
             dispatch_uid
                 An identifier used to uniquely identify a particular instance of
                 a receiver. This will usually be a string, though it may be
                 anything hashable.
         """
         from django.conf import settings
-        
+
         # If DEBUG is on, check that we got a good receiver
         if settings.DEBUG:
             import inspect
             assert callable(receiver), "Signal receivers must be callable."
-            
+
             # Check for **kwargs
             # Not all callables are inspectable with getargspec, so we'll
             # try a couple different ways but in the end fall back on assuming
@@ -90,22 +92,19 @@
             if argspec:
                 assert argspec[2] is not None, \
                     "Signal receivers must accept keyword arguments (**kwargs)."
-        
-        if dispatch_uid:
-            lookup_key = (dispatch_uid, _make_id(sender))
-        else:
-            lookup_key = (_make_id(receiver), _make_id(sender))
 
+        lookup_key = dispatch_uid or _make_id(receiver)
+
         if weak:
             receiver = saferef.safeRef(receiver, onDelete=self._remove_receiver)
 
         self.lock.acquire()
         try:
-            for r_key, _ in self.receivers:
-                if r_key == lookup_key:
-                    break
-            else:
-                self.receivers.append((lookup_key, receiver))
+            # Should really use OrderedDict in python 2.7
+            registered_keys = (rec[1] for rec in self.receivers[sender])
+            if lookup_key not in registered_keys:
+                self.receivers[sender].append((lookup_key, receiver, self.connect_index))
+                self.connect_index += 1
         finally:
             self.lock.release()
 
@@ -115,33 +114,29 @@
 
         If weak references are used, disconnect need not be called. The receiver
         will be remove from dispatch automatically.
-    
+
         Arguments:
-        
+
             receiver
                 The registered receiver to disconnect. May be none if
                 dispatch_uid is specified.
-            
+
             sender
                 The registered sender to disconnect
-            
+
             weak
                 The weakref state to disconnect
-            
+
             dispatch_uid
                 the unique identifier of the receiver to disconnect
         """
-        if dispatch_uid:
-            lookup_key = (dispatch_uid, _make_id(sender))
-        else:
-            lookup_key = (_make_id(receiver), _make_id(sender))
-        
+        lookup_key = dispatch_uid or _make_id(receiver)
+
         self.lock.acquire()
         try:
-            for index in xrange(len(self.receivers)):
-                (r_key, _) = self.receivers[index]
-                if r_key == lookup_key:
-                    del self.receivers[index]
+            for index, (key, _, _) in enumerate(self.receivers[sender]):
+                if key == lookup_key:
+                    del self.receivers[sender][index]
                     break
         finally:
             self.lock.release()
@@ -155,20 +150,18 @@
         receivers called if a raises an error.
 
         Arguments:
-        
+
             sender
                 The sender of the signal Either a specific object or None.
-    
+
             named
                 Named arguments which will be passed to receivers.
 
         Returns a list of tuple pairs [(receiver, response), ... ].
         """
         responses = []
-        if not self.receivers:
-            return responses
 
-        for receiver in self._live_receivers(_make_id(sender)):
+        for receiver in self._live_receivers(sender):
             response = receiver(signal=self, sender=sender, **named)
             responses.append((receiver, response))
         return responses
@@ -178,7 +171,7 @@
         Send signal from sender to all connected receivers catching errors.
 
         Arguments:
-        
+
             sender
                 The sender of the signal. Can be any python object (normally one
                 registered with a connect if you actually want something to
@@ -197,12 +190,10 @@
         receiver.
         """
         responses = []
-        if not self.receivers:
-            return responses
 
         # Call each receiver with whatever arguments it can accept.
         # Return a list of tuple pairs [(receiver, response), ... ].
-        for receiver in self._live_receivers(_make_id(sender)):
+        for receiver in self._live_receivers(sender):
             try:
                 response = receiver(signal=self, sender=sender, **named)
             except Exception, err:
@@ -211,45 +202,57 @@
                 responses.append((receiver, response))
         return responses
 
-    def _live_receivers(self, senderkey):
+    def _live_receivers(self, sender):
         """
-        Filter sequence of receivers to get resolved, live receivers.
+        Construct sequence of receivers to get resolved, live receivers.
 
         This checks for weak references and resolves them, then returning only
         live receivers.
         """
-        none_senderkey = _make_id(None)
-        receivers = []
+        def combine(a, b):
+            if not a:
+                return b
+            elif not b:
+                return a
 
-        for (receiverkey, r_senderkey), receiver in self.receivers:
-            if r_senderkey == none_senderkey or r_senderkey == senderkey:
+            a_index = 0
+            b_index = 0
+            while a_index < len(a) and b_index < len(b):
+                if a[a_index][2] < b[b_index][2]:
+                    yield a[a_index][1]
+                    a_index += 1
+                else:
+                    yield b[b_index][1]
+                    b_index += 1
+            for i in range(a_index, len(a)):
+                yield a[i][1]
+            for i in range(b_index, len(b)):
+                yield b[i][1]
+
+        def deref(receivers):
+            for receiver in receivers:
                 if isinstance(receiver, WEAKREF_TYPES):
                     # Dereference the weak reference.
                     receiver = receiver()
                     if receiver is not None:
-                        receivers.append(receiver)
+                        yield receiver
                 else:
-                    receivers.append(receiver)
-        return receivers
+                    yield receiver
 
-    def _remove_receiver(self, receiver):
+        receivers = combine(self.receivers[None], self.receivers[sender])
+        return list(deref(receivers))
+
+    def _remove_receiver(self, dead_receiver):
         """
         Remove dead receivers from connections.
         """
 
         self.lock.acquire()
         try:
-            to_remove = []
-            for key, connected_receiver in self.receivers:
-                if connected_receiver == receiver:
-                    to_remove.append(key)
-            for key in to_remove:
-                last_idx = len(self.receivers) - 1
-                # enumerate in reverse order so that indexes are valid even
-                # after we delete some items
-                for idx, (r_key, _) in enumerate(reversed(self.receivers)):
-                    if r_key == key:
-                        del self.receivers[last_idx-idx]
+            for receivers in self.receivers.values():
+                for index, (key, receiver, _) in enumerate(receivers):
+                    if receiver == dead_receiver:
+                        del receivers[index]
         finally:
             self.lock.release()
 
