from django.db.models import (
    TextField, IntegerField, FloatField, Lookup, Extract)


# We are going to play with casts, so implement a couple of those first.
class Cast(Extract):
    def as_sql(self, qn, connection):
        sql, params = qn.compile(self.lhs)
        # Drop the as_ prefix
        cast_type = self.lookup_name[3:]
        return '(%s)::%s' % (sql, cast_type), params


class IntCast(Cast):
    lookup_name = 'as_int'
    output_type = IntegerField()


class FloatCast(Cast):
    lookup_name = 'as_float'
    output_type = FloatField()


# For fun: lets implement nestable hstore
class HStoreCast(Cast):
    lookup_name = 'as_hstore'

    # Need property due to circular dependency
    @property
    def output_type(self):
        return HStoreField()


class CastableText(TextField):
    pass
CastableText.register_lookup(IntCast)
CastableText.register_lookup(FloatCast)
CastableText.register_lookup(HStoreCast)


# This one implements "get value of key X from the hstore"
class HStoreExtract(Extract):
    output_type = CastableText()

    def as_sql(self, qn, connection):
        lhs, params = qn.compile(self.lhs)
        params.append(self.init_lookups[0])
        return '%s -> %%s' % (lhs), params


# This one implements "does hstore contain key X"
class HStoreContains(Lookup):
    lookup_name = 'hcontains'

    def as_sql(self, qn, connection):
        lhs, params = self.process_lhs(qn, connection)
        rhs, rhs_params = self.process_rhs(qn, connection)
        params.extend(rhs_params)
        return '%s ? %s' % (lhs, rhs), params


# Main HStoreField
class HStoreField(TextField):
    # Following two methods needed for createdb & save support.
    def db_type(self, connection):
        return 'hstore'

    def get_db_prep_save(self, value, connection):
        data = []
        for k, v in value.items():
            data.append('%s=>%s' % (k, v))
        return ','.join(data)

    def get_lookup(self, lookup):
        found_lookup = None
        # It is possible to rename Django's own lookups so that conflicts
        # between common names (contains, exact for example) can be avoided.
        if lookup.startswith('dj_'):
            # If it started with 'dj_' use Django's own lookups
            found_lookup = super(HStoreField, self).get_lookup(lookup[3:])
        if found_lookup is None:
            return HStoreExtract
        else:
            return found_lookup
HStoreField.register_lookup(HStoreContains)
