from django.test import TestCase
from proxy_bug.models import Image, File, Photo, FooFile, FooImage, FooPhoto


class ProxyDeleteImageTest(TestCase):
    '''
    Tests on_delete behaviour for proxy models. Deleting the *proxy*
    instance bubbles through to its non-proxy and *all* referring objects
    are deleted.
    '''

    def setUp(self):
        # Create an Image
        test_image = Image()
        test_image.save()
        foo_image = FooImage(my_image=test_image)
        foo_image.save()

        # Get the Image instance as a File
        test_file = File.objects.all()[0]
        foo_file = FooFile(my_file=test_file)
        foo_file.save()

    def test_delete(self):
        self.assertTrue(Image.objects.all().delete() is None)
        # An Image deletion == File deletion
        self.assertEqual(len(Image.objects.all()), 0)
        self.assertEqual(len(File.objects.all()), 0)
        # The Image deletion cascaded and *all* references to it are deleted.
        self.assertEqual(len(FooImage.objects.all()), 0)
        self.assertEqual(len(FooFile.objects.all()), 0)


class ProxyDeletePhotoTest(ProxyDeleteImageTest):
    '''
    Tests on_delete behaviour for proxy-of-proxy models. Deleting the *proxy*
    instance should bubble through to its proxy and non-proxy variants.
    Deleting *all* referring objects. For some reason it seems that the 
    intermediate proxy model isn't cleaned up.
    '''

    def setUp(self):
        # Create the Image, FooImage and FooFile instances
        super(ProxyDeletePhotoTest, self).setUp()
        # Get the Image as a Photo
        test_photo = Photo.objects.all()[0]
        foo_photo = FooPhoto(my_photo=test_photo)
        foo_photo.save()


    def test_delete(self):
        self.assertTrue(Photo.objects.all().delete() is None)
        # A Photo deletion == Image deletion == File deletion
        self.assertEqual(len(Photo.objects.all()), 0)
        self.assertEqual(len(Image.objects.all()), 0)
        self.assertEqual(len(File.objects.all()), 0)
        # The Photo deletion should have cascaded and deleted *all*
        # references to it.
        self.assertEqual(len(FooPhoto.objects.all()), 0)
        self.assertEqual(len(FooFile.objects.all()), 0)
        self.assertEqual(len(FooImage.objects.all()), 0)


class ProxyDeleteFileTest(ProxyDeleteImageTest):
    '''
    Tests on_delete cascade behaviour for proxy models. Deleting the
    *non-proxy* instance of a model should somehow notify it's proxy.
    Currently it doesn't, making this test fail.
    '''

    def test_delete(self):
        # This should *not* raise an IntegrityError on databases that support 
        # FK constraints.
        self.assertTrue(File.objects.all().delete() is None)
        # A File deletion == Image deletion
        self.assertEqual(len(File.objects.all()), 0)
        self.assertEqual(len(Image.objects.all()), 0)
        # The File deletion should have cascaded and deleted *all* references
        # to it.
        self.assertEqual(len(FooFile.objects.all()), 0)
        self.assertEqual(len(FooImage.objects.all()), 0)
