import datetime
from django.db.transaction import commit_on_success
import time
from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.utils.safestring import mark_safe

from django.contrib.contenttypes.models import ContentType

from addressbook.models import Address
from common.models import File, AnnotedSchema
from machines.models import Machine
from stock.models import SparePart, Location

REPETITION_UNITS = (('H', _('hour(s)')),
                    ('D', _('day(s)')),
                    ('W', _('week(s)')),
                    ('M', _('month(s)')))

OPERATION_MAINTENANCE = 'maintenance'
OPERATION_FAILURE = 'failure'
OPERATION_METHOD = 'method'
OPERATION_CUSTOM = 'custom'

SEARCH_OPERATION_TYPES = ((OPERATION_MAINTENANCE, _('Maintenance')),
                         (OPERATION_FAILURE, _('Failure')),
                         (OPERATION_METHOD, _('Method')),
                         (OPERATION_CUSTOM, _('Custom sheet')))

class OperationType(models.Model):
    name = models.CharField(_('Name'), max_length=30, unique=True)

    class Meta:
        db_table = 'operation_type'

class StaffType(models.Model):
    name = models.CharField(_('Name'), max_length=30, unique=True)

    class Meta:
        db_table = 'staff_type'

class Operation(models.Model):
    name = models.CharField(_('name'), max_length=150)
    type = models.ForeignKey(OperationType, verbose_name=_('Action type'), null=True, blank=True)
    staff = models.ForeignKey(StaffType, verbose_name=_('Staff type'), null=True, blank=True)
    description = models.TextField(_('Description'))
    files = models.ManyToManyField(File, blank=True, related_name='maintenance', verbose_name=_('Files'))
    content_type = models.ForeignKey(ContentType, editable=False, null=True)
    # maintenance and failure
    annoted_schema = models.ForeignKey(AnnotedSchema, verbose_name=_('Annoted schema'), blank=True, null=True)

    class Meta:
        db_table = 'operation'

class SparePartWithdrawal(models.Model):
    """
    it's a copy of stocks.RollingOfStock
    it's used to define rolling of stock for operation which
    will be created when the event occurs.
    A separate model is used to simplify complex query for
    rolling of stock.
    """
    spare_part = models.ForeignKey(SparePart, verbose_name=_('Spare part'), related_name="withdrawals")
    quantity = models.IntegerField(_('Quantity'))
    operation = models.ForeignKey(Operation, related_name="spare_parts", verbose_name=_('Operation'))
    # Presence of location in checked in clean() of form
    location = models.ForeignKey(Location, verbose_name=_('Location'), blank=True, null=True)

    class Meta:
        db_table = 'spare_part_withdrawal'

class MaintenanceNature(models.Model):
    name = models.CharField(_('Name'), max_length=30, unique=True)

    class Meta:
        db_table = 'maintenance_nature'
        verbose_name = _('Maintenance nature')
        verbose_name_plural = _('Maintenance natures')

class Maintenance(Operation):
    nature = models.ForeignKey(MaintenanceNature, verbose_name=_('Action nature'))
    machine = models.ForeignKey(Machine, verbose_name=_('Equipment'), related_name='maintenances')
    duration_days = models.PositiveIntegerField(_('Duration'), help_text=_('days'), blank=True, null=True)
    duration_hours = models.PositiveIntegerField(help_text=_('hours'), blank=True, null=True)
    duration_minutes = models.PositiveIntegerField(help_text=_('minutes'), blank=True, null=True)
    start_date = models.DateField(_('Start date'), blank=True, null=True)
    end_date = models.DateField(_('End date'), blank=True, null=True)
    repetition_value = models.PositiveIntegerField(_('Repetition'), blank=True, null=True)
    repetition_unit = models.CharField(max_length=1, choices=REPETITION_UNITS, default='M')
    tolerance_value = models.PositiveIntegerField(_('Tolerance'), blank=True, null=True)
    tolerance_unit = models.CharField(max_length=1, choices=REPETITION_UNITS, default='D')
    people_to_notify = models.ManyToManyField(Address, blank=True, null=True, verbose_name=_('People to notify'))
    notification_delay = models.IntegerField(_('Notification delay'), null=True, blank=True)

    class Meta:
        db_table = 'maintenance'

class MaintenanceOccurence(models.Model):
    datetime = models.DateTimeField(_('Date'))
    maintenance = models.ForeignKey(Maintenance, related_name='occurences', verbose_name=_('Maintenance'))

    class Meta:
        db_table = 'maintenance_occurence'

class Failure(Operation):
    machine = models.ForeignKey(Machine, verbose_name=_('Equipment'), related_name='failures')
    generic = models.BooleanField(_('Generic failure sheet for machine'))

    class Meta:
        db_table = 'failure'

class MethodType(models.Model):
    name = models.CharField(_('Name'), max_length=30, unique=True)

    class Meta:
        db_table = 'method_type'

class Method(Operation):
    machine = models.ForeignKey(Machine, verbose_name=_('Equipment'))
    method_type = models.ForeignKey(MethodType, verbose_name=_('Method type'))
    scheduled_datetime = models.DateTimeField(_('Scheduled date'), null=True, blank=True)
    people_to_notify = models.ManyToManyField(Address, blank=True, null=True, verbose_name=_('People to notify'))
    notification_delay = models.IntegerField(_('Notification delay'), null=True, blank=True)

    class Meta:
        db_table = 'method'

class Custom(Operation):
    machine = models.ForeignKey(Machine, verbose_name=_('Equipment'))

    class Meta:
        db_table = 'custom'
