import datetime

from django.core.mail import SMTPConnection, EmailMessage
from django.core.urlresolvers import reverse
from django.contrib.sites.models import Site
from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.utils.translation import ugettext, ugettext_lazy as _
from django.utils.safestring import mark_safe

from common.templatetags.date_extras import date_l10n
from common.models import Configuration, File, signal_event_invalidating_statistics_saved, AnnotedSchema
from operations.models import Maintenance, Failure, Method, Custom, Operation, OPERATION_MAINTENANCE,\
OPERATION_FAILURE, OPERATION_METHOD, OPERATION_CUSTOM
from organization.models import Operator
from machines.models import Machine, Channel
from mycalendar.models import Period
from mycalendar.views import get_periods as mycalendar_get_periods
from stock.models import RollingOfStock

EVENT = 'event'
EVENT_MAINTENANCE = 'eventmaintenance'
EVENT_FAILURE = 'eventfailure'
EVENT_MAKINGORDER = 'eventmakingorder'
EVENT_METHOD = 'eventmethod'
EVENT_CUSTOM = 'eventcustom'

SEARCH_EVENT_TYPES = (
    (EVENT_MAINTENANCE, _('Maintenance')),
    (EVENT_FAILURE, _('Failure')),
    (EVENT_MAKINGORDER, _('Making order')),
    (EVENT_METHOD, _('Method')),
    (EVENT_CUSTOM, _('Custom event'))
)

EVENT_TO_OPERATION_TYPES = {
    EVENT_MAINTENANCE : OPERATION_MAINTENANCE,
    EVENT_FAILURE : OPERATION_FAILURE,
    EVENT_MAKINGORDER : '',
    EVENT_METHOD : OPERATION_METHOD,
    EVENT_CUSTOM : OPERATION_CUSTOM
}

class SimplePeriod(models.Model):
    begin_datetime = models.DateTimeField(_('Begin'))
    end_datetime = models.DateTimeField(_('End'))
    class Meta:
        db_table = 'simple_period'
        verbose_name = _('Simple period')
        verbose_name_plural = _('Simple periods')

    def __unicode__(self):
        return unicode(_(u"From %(begin_datetime)s to %(end_datetime)s") % {'begin_datetime': date_l10n(self.begin_datetime), 'end_datetime': date_l10n(self.end_datetime)})

class Event(models.Model):
    machine = models.ForeignKey(Machine, related_name='events', verbose_name=_('Equipment'))
    begin_datetime = models.DateTimeField(_('Begin'), blank=True, null=True, db_index=True)
    end_datetime = models.DateTimeField(_('End'), blank=True, null=True, db_index=True)
    content_type = models.ForeignKey(ContentType, editable=False, null=True)
    operators = models.ManyToManyField(Operator, related_name='events',
                                       verbose_name=_('Involved operators'), blank=True, null=True)
    files = models.ManyToManyField(File, blank=True, related_name='event', verbose_name=_('Files'))
    worked_periods = models.ManyToManyField(SimplePeriod, blank=True, null=True, related_name='event', verbose_name=_('Worked periods'))
    objects = EventManager()
    annoted_schema = models.ForeignKey(AnnotedSchema, verbose_name=_('Annoted schema'), blank=True, null=True)

    class Meta:
        db_table = 'event'

    @staticmethod
    def get_associated_operation_cls():
        return Operation

class EventChannel(Event):
    channel = models.ForeignKey(Channel, related_name='events', verbose_name=_('Channel'))

    class Meta:
        db_table = 'event_channel'

    @staticmethod
    def get_associated_operation_cls():
        return None

class EventFailure(EventChannel):
    description = models.TextField(_('Description'), blank=True, null=True)
    failure = models.ForeignKey(Failure, related_name='events', verbose_name=_('Failure'))
    rolling_of_stocks = models.ManyToManyField(RollingOfStock,
                                               verbose_name=_('Rollings of stock'))
    cost = models.FloatField(_('Cost'), blank=True, null=True)

    class Meta:
        db_table = 'event_failure'

    @staticmethod
    def get_associated_operation_cls():
        return Failure

class EventMaintenance(Event):
    maintenance = models.ForeignKey(Maintenance, related_name='events', verbose_name=_('Maintenance'))
    description = models.TextField(_('Description'), blank=True, null=True)
    rolling_of_stocks = models.ManyToManyField(RollingOfStock, verbose_name=_('Rollings of stock'))
    cost = models.FloatField(_('Cost'), blank=True, null=True)


    class Meta:
        db_table = 'event_maintenance'

    @staticmethod
    def get_associated_operation_cls():
        return Maintenance

class EventMakingOrder(Event):
    barcode = models.CharField(_('Barcode'), max_length=12)
    quality_rate = models.DecimalField(_('Quality rate'), max_digits=4, decimal_places=2, blank=True, null=True)
    theorical_cycle_time = models.IntegerField(_('Theorical cycle time (in seconds)'), blank=True, null=True)
    cycle_count_done = models.IntegerField(_('Cycle count done'), blank=True, null=True)
    theorical_cycle_count = models.IntegerField(_('Theorical cycle count'), blank=True, null=True)
    reference_cycle_time = models.IntegerField(_('Reference cycle time (in seconds)'), blank=True, null=True)
    theorical_intercycle_time = models.IntegerField(_('Theorical intercycle time (in seconds)'), blank=True, null=True)
    computed_intercycle_time = models.IntegerField(_('Computed intercycle time (in seconds)'), blank=True, null=True)

    class Meta:
        db_table = 'event_making_order'

    @staticmethod
    def get_associated_operation_cls():
        return None

class EventMethod(Event):
    method = models.ForeignKey(Method, verbose_name=_("Method"), related_name="events", blank=True, null=True)
    description = models.TextField(_('Description'), blank=True, null=True)
    rolling_of_stocks = models.ManyToManyField(RollingOfStock, verbose_name=_('Rollings of stock'))
    cost = models.FloatField(_('Cost'), blank=True, null=True)

    class Meta:
        db_table = 'event_method'

    @staticmethod
    def get_associated_operation_cls():
        return Method

class EventCustom(Event):
    custom = models.ForeignKey(Custom, verbose_name=_("Custom sheet"), related_name="events", blank=True, null=True)
    description = models.TextField(_('Description'), blank=True, null=True)

    class Meta:
        db_table = 'event_custom'

    @staticmethod
    def get_associated_operation_cls():
        return Custom
