# test/othertests/fields.py
from django.core import validators
from django.db.models.fields import *
import random
import unittest
import datetime, time
from decimal import Decimal
import re

errors = 0
failures = 0
def run_test(test):
    if __name__ == '__main__': return
    global errors, failures
    suite = unittest.makeSuite(test)
    ret = unittest.TextTestRunner().run(suite)
    errors += len(ret.errors)
    failures += len(ret.failures)
#

# TODO: AutoField
class TestAutoField(unittest.TestCase):
    def setUp(self):
        self.klass = AutoField
    #
#
run_test(TestAutoField)

class TestBooleanField(unittest.TestCase):
    def setUp(self):
        self.klass = BooleanField
    #
    def test_create(self):
        """ BooleanField:: create """
        try:
            self.klass(null=True)
            self.assert_('No Assertion error for null=True')
        except AssertionError:
            pass
    #

    def test_conversion(self):
        """ BooleanField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('t'), True)
        self.assertEqual(cf.to_python('f'), False)
        self.assertEqual(cf.to_python(True), True)
        self.assertEqual(cf.to_python(False), False)
        self.assertEqual(cf.to_python(''), False)
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        self.assertRaises(validators.ValidationError, cf.to_python, 'asd')
        self.assertRaises(validators.ValidationError, cf.to_python, 10)
    #
#
run_test(TestBooleanField)

class TestNullBooleanField(unittest.TestCase):
    def setUp(self):
        self.klass = NullBooleanField
    #

    def test_conversion(self):
        """ NullBooleanField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('t'), True)
        self.assertEqual(cf.to_python('f'), False)
        self.assertEqual(cf.to_python(True), True)
        self.assertEqual(cf.to_python(False), False)
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
        self.assertRaises(validators.ValidationError, cf.to_python, 'asd')
        self.assertRaises(validators.ValidationError, cf.to_python, 10)
    #
#
run_test(TestNullBooleanField)

class TestCommaSeparatedIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = CommaSeparatedIntegerField
    #
    
    def test_validate_maxlength(self):
        """ CommaSeparatedIntegerField:: maxlength validation """
        cf = self.klass(maxlength=2)
        data = [1, 2, 3]
        self.assertRaises(validators.ValidationError, cf.validate, data, {})
        cf = self.klass(maxlength=3)
        cf.validate(data, {})
    #

    def test_conversion(self):
        """ CommaSeparatedIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1, 2, 3'), [1, 2, 3])
        self.assertEqual(cf.to_python(''), [])
        self.assertEqual(cf.to_python(None), [])
        self.assertRaises(validators.ValidationError, cf.validate, '1, a, 3', {})
    #
#
run_test(TestCommaSeparatedIntegerField)

class TestCharField(unittest.TestCase):
    def setUp(self):
        self.klass = CharField
    #

    def test_conversion(self):
        """ CharField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python(1), '1')
        self.assertEqual(cf.to_python('10'), '10')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #

    # I don't need to test null/blank validation, to_pyhon takes care of that
    def test_validate_maxlength(self):
        """ CharField:: maxlength validation """
        cf = self.klass(maxlength=10)
        self.assertRaises(validators.ValidationError, cf.validate, 'a' * 11, {})
        cf.validate('b' * 10, {})
    #
#
run_test(TestCharField)

class TestDateField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = DateField
        self.data = '2006-10-15'
        self.ret = datetime.date(*time.strptime(self.data, '%Y-%m-%d')[:3])
    #

    def test_conversion(self):
        """ DateField:: conversion """
        cf = self.klass()
        self.assert_(isinstance(cf.to_python(self.data), datetime.date))
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
        # TODO: support for other date formats??
    #
#
run_test(TestDateField)

class TestDateTimeField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = DateTimeField
        self.data = '2006-10-15 13:15:23'
        self.ret = datetime.datetime(*time.strptime(self.data, '%Y-%m-%d %H:%M:%S')[:6])
    #

    def test_conversion(self):
        """ DateTimeField:: conversion """
        cf = self.klass()
        self.assert_(isinstance(cf.to_python(self.data), datetime.date))
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
        # TODO: support for other date formats??
    #
#
run_test(TestDateTimeField)


class TestEmailField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = EmailField
    #

    def test_conversion(self):
        """ EmailField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('a@bc.de'), 'a@bc.de')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #

    def test_validate(self):
        """ EmailField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, 'a-b-c', {})
        cf.validate('a@bc.de', {})
    #
#
run_test(TestEmailField)

# TODO: FileField
class TestFileField(unittest.TestCase):
    def setUp(self):
        self.klass = FileField
    #
#
run_test(TestFileField)
# TODO: FilePathField
class TestFilePathField(unittest.TestCase):
    def setUp(self):
        self.klass = FilePathField
    #
#
run_test(TestFilePathField)

class TestFloatField(unittest.TestCase):
    def setUp(self):
        self.klass = FloatField
    #
    def test_conversion(self):
        """ FloatField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1.0'), Decimal('1.0'))
    #

    def test_validate(self):
        """ FloatField:: validate """
        cf = self.klass(max_digits=2)
        self.assertRaises(validators.ValidationError, cf.validate, Decimal('132.21'), {})
        cf.validate(Decimal('12.1223'), {})
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestFloatField)
# TODO: ImageField
class TestImageField(unittest.TestCase):
    def setUp(self):
        self.klass = ImageField
    #
#
run_test(TestImageField)

class TestIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = IntegerField
    #
    def test_conversion(self):
        """ IntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestIntegerField)

class TestIPAddressField(unittest.TestCase):
    def setUp(self):
        self.klass = IPAddressField
    #
    def test_validate(self):
        """ IPAddressField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, '12.13.4.257', {})
        cf.validate('1.2.3.4', {})
    #

    def test_conversion(self):
        """ IPAddressField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1.2.3.4'), '1.2.3.4')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestIPAddressField)
# TODO: PhoneNumberField
# XXX Maybe this is more for contrib than trunk?
class TestPhoneNumberField(unittest.TestCase):
    def setUp(self):
        self.klass = PhoneNumberField
    #
#
run_test(TestPhoneNumberField)

class TestPositiveIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = PositiveIntegerField
    #
    def test_conversion(self):
        """ PositiveIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
    #

    def test_validate(self):
        """ PositiveIntegerField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, -1, {})
        cf.validate(1, {})
    #
#
run_test(TestPositiveIntegerField)

class TestPositiveSmallIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = PositiveSmallIntegerField
    #

    def test_conversion(self):
        """ PositiveSmallIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
    #

    def test_validate(self):
        """ PositiveSmallIntegerField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, -1, {})
        cf.validate(1, {})
    #
#
run_test(TestPositiveSmallIntegerField)

class TestSlugField(unittest.TestCase):
    def setUp(self):
        self.klass = SlugField
    #
    def test_validate(self):
        """ SlugField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, 'asAD d-ad_ad', {})
        self.assertRaises(validators.ValidationError, cf.validate, '1 bad-start', {})
        self.assertRaises(validators.ValidationError, cf.validate, 'bad-end 2', {})
        cf.validate('valid-slug_', {})
    #

    def test_conversion(self):
        """ SlugField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('valid-slug'), 'valid-slug')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestSlugField)

class TestSmallIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = SmallIntegerField
    #

    def test_conversion(self):
        """ SmallIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestSmallIntegerField)

class TestTextField(unittest.TestCase):
    def setUp(self):
        self.klass = TextField
    #

    def test_conversion(self):
        """ TextField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('12sa asd asd azxc'), '12sa asd asd azxc')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestTextField)

class TestTimeField(unittest.TestCase):
    def setUp(self):
        self.klass = TimeField
        self.data = '13:45:16'
        self.ret = time.strptime(self.data, '%H:%M:%S')
    #

    def test_conversion(self):
        """ TimeField:: conversion """
        cf = self.klass()
        ret = cf.to_python(self.data)
        self.assert_(isinstance(ret, datetime.time), '%s != %s' % (type(ret), datetime.time))
        self.assertEqual(cf.to_python(''), None)
        self.assertEqual(cf.to_python(None), None)
        # TODO: test other time formats and TZ info
    #
#
run_test(TestTimeField)

class TestURLField(unittest.TestCase):
    def setUp(self):
        self.klass = URLField
    #
    
    def test_conversion(self):
        """ URLField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('www.djangoproject.com'), 'www.djangoproject.com')
        self.assertEqual(cf.to_python(''), '')
        self.assertEqual(cf.to_python(None), None)
    #

    def test_validate(self):
        """ URLField:: validate (this can fail if you don't have internet connection)"""
        cf = self.klass(verify_exists=False)
        self.assertRaises(validators.ValidationError, cf.validate, 'this.not.exists', {})
        cf.validate('http://www.djangoproject.com', {})
    #

    def test_validate_exists(self):
        """ URLField:: validate """
        cf = self.klass(verify_exists=True)
        self.assertRaises(validators.ValidationError, cf.validate, 'http://this.not.exists', {})
        cf.validate('http://www.djangoproject.com', {})
    #
#
run_test(TestURLField)

# TODO: USStateField
# XXX Maybe this is more for contrib than trunk?
class TestUSStateField(unittest.TestCase):
    def setUp(self):
        self.klass = USStateField
    #
#
run_test(TestUSStateField)
# TODO: XMLField
class TestXMLField(unittest.TestCase):
    def setUp(self):
        self.klass = XMLField
    #
#
run_test(TestXMLField)

# THE BIG TODO
# TODO: OrderingField
# TODO: ForeignKey
# TODO: OneToOneField
# TODO: ManyToManyField

if __name__ != '__main__':
    # for django runtests.py
    if errors or failures:
        assert False, 'some tests are failed (E: %d, F:%d)' % (errors, failures)
else:
    unittest.main()