# test/othertests/fields.py
from django.core import validators
from django.db.models.fields import *
import random
import unittest
import datetime, time
from decimal import Decimal
import re

errors = 0
failures = 0
def run_test(test):
    if __name__ == '__main__': return
    global errors, failures
    suite = unittest.makeSuite(test)
    ret = unittest.TextTestRunner().run(suite)
    errors += len(ret.errors)
    failures += len(ret.failures)
#

class TestAutoField(unittest.TestCase):
    def setUp(self):
        self.klass = AutoField
    #
    def test_create(self):
        """ AutoField:: create """
        self.assertRaises(AssertionError, self.klass)
        self.klass(primary_key=True)
        try:
            self.klass(primary_key=True, blank=True)
            self.assert_('No Assertion error for blank=True')
        except AssertionError:
            pass
        try:
            self.klass(primary_key=True, null=True)
            self.assert_('No Assertion error for null=True')
        except AssertionError:
            pass
    #

    def test_conversion(self):
        """ AutoField:: conversion """
        cf = self.klass(primary_key=True)
        self.assertEqual(cf.to_python(1), 1)
        self.assertEqual(cf.to_python('10'), 10)
    #
#
run_test(TestAutoField)

class TestBooleanField(unittest.TestCase):
    def setUp(self):
        self.klass = BooleanField
    #
    def test_create(self):
        """ BooleanField:: create """
        try:
            self.klass(null=True)
            self.assert_('No Assertion error for null=True')
        except AssertionError:
            pass
    #

    def test_conversion(self):
        """ BooleanField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('t'), True)
        self.assertEqual(cf.to_python('f'), False)
        self.assertEqual(cf.to_python(True), True)
        self.assertEqual(cf.to_python(False), False)
        # XXX currently BooleanField don't accept integers as valid values which IMHO it should
        #self.assertEqual(cf.to_python(''), False)
        #self.assertEqual(cf.to_python(4023), True)
        #self.assertEqual(cf.to_python(-4023), True)
        #self.assertEqual(cf.to_python(0), False)
    #
#
run_test(TestBooleanField)

class TestNullBooleanField(unittest.TestCase):
    def setUp(self):
        self.klass = NullBooleanField
    #

    def test_conversion_null(self):
        """ NullBooleanField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #

    def test_conversion(self):
        """ NullBooleanField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('t'), True)
        self.assertEqual(cf.to_python('f'), False)
        self.assertEqual(cf.to_python(True), True)
        self.assertEqual(cf.to_python(False), False)
        self.assertEqual(cf.to_python(None), None)
        # XXX currently BooleanField don't accept integers as valid values which IMHO it should
        #self.assertEqual(cf.to_python(''), False)
        #self.assertEqual(cf.to_python(4023), True)
        #self.assertEqual(cf.to_python(-4023), True)
        #self.assertEqual(cf.to_python(0), False)
    #
#
run_test(TestNullBooleanField)

class TestCommaSeparatedIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = CommaSeparatedIntegerField
    #
    
    def test_conversion_blank(self):
        """ CommaSeparatedIntegerField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ CommaSeparatedIntegerField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), [])
    #
    # I don't need to test null/blank validation, to_pyhon takes care of that
    def test_validate_maxlength(self):
        """ CommaSeparatedIntegerField:: maxlength validation """
        cf = self.klass(maxlength=2)
        data = [1, 2, 3]
        self.assertRaises(validators.ValidationError, cf.validate, data, {})
        cf = self.klass(maxlength=3)
        cf.validate(data, {})
    #

    def test_conversion(self):
        """ CommaSeparatedIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1, 2, 3'), [1, 2, 3])
        self.assertRaises(validators.ValidationError, cf.validate, '1, a, 3', {})
    #
#
run_test(TestCommaSeparatedIntegerField)

class TestCharField(unittest.TestCase):
    def setUp(self):
        self.klass = CharField
    #

    def test_conversion(self):
        """ CharField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python(1), '1')
        self.assertEqual(cf.to_python('10'), '10')
    #

    def test_conversion_blank(self):
        """ CharField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ CharField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
    # I don't need to test null/blank validation, to_pyhon takes care of that
    def test_validate_maxlength(self):
        """ CharField:: maxlength validation """
        cf = self.klass(maxlength=10)
        self.assertRaises(validators.ValidationError, cf.validate, 'a' * 11, {})
        cf.validate('b'*10, {})
    #
#
run_test(TestCharField)

class TestDateField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = DateField
        self.data = '2006-10-15'
        self.ret = datetime.date(*time.strptime(self.data, '%Y-%m-%d')[:3])
    #

    def test_conversion(self):
        """ DateField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python(self.data), self.ret)
        # TODO: support for other date formats??
    #

    def test_conversion_null(self):
        """ DateField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestDateField)

class TestDateTimeField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = DateTimeField
        self.data = '2006-10-15 13:15:23'
        self.ret = datetime.datetime(*time.strptime(self.data, '%Y-%m-%d %H:%M:%S')[:3])
    #

    def test_conversion(self):
        """ DateTimeField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python(self.data), self.ret)
        # TODO: support for other date formats??
    #

    def test_conversion_null(self):
        """ DateTimeField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestDateTimeField)


class TestEmailField(unittest.TestCase):
    # TODO: auto_now stuff tests
    def setUp(self):
        self.klass = EmailField
    #

    def test_conversion(self):
        """ EmailField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('a@bc.de'), 'a@bc.de')
    #

    def test_validate(self):
        """ EmailField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, 'a-b-c', {})
        cf.validate('a@bc.de', {})
    #

    def test_conversion_blank(self):
        """ EmailField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ EmailField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestEmailField)

# TODO: FileField
class TestFileField(unittest.TestCase):
    def setUp(self):
        self.klass = FileField
    #
#
run_test(TestFileField)
# TODO: FilePathField
class TestFilePathField(unittest.TestCase):
    def setUp(self):
        self.klass = FilePathField
    #
#
run_test(TestFilePathField)

class TestFloatField(unittest.TestCase):
    def setUp(self):
        self.klass = FloatField
    #
    def test_conversion(self):
        """ FloatField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1.0'), Decimal('1.0'))
    #

    def test_validate(self):
        """ FloatField:: validate """
        cf = self.klass(max_digits=2)
        self.assertRaises(validators.ValidationError, cf.validate, Decimal('132.21'), {})
        cf.validate(Decimal('12.1223'), {})
    #

    def test_conversion_blank(self):
        """ FloatField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ FloatField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestFloatField)
# TODO: ImageField
class TestImageField(unittest.TestCase):
    def setUp(self):
        self.klass = ImageField
    #
#
run_test(TestImageField)

class TestIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = IntegerField
    #
    def test_conversion(self):
        """ IntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
    #

    def test_conversion_blank(self):
        """ IntegerField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ IntegerField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestIntegerField)
# TODO: IPAddressField
class TestIPAddressField(unittest.TestCase):
    def setUp(self):
        self.klass = IPAddressField
    #
    def test_validate(self):
        """ IPAddressField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, '12.13.4.257', {})
        cf.validate('1.2.3.4', {})
    #

    def test_conversion(self):
        """ IPAddressField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('1.2.3.4'), '1.2.3.4')
    #

    def test_conversion_blank(self):
        """ IPAddressField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ IPAddressField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestIPAddressField)
# TODO: PhoneNumberField
# XXX Maybe this is more for contrib than trunk?
class TestPhoneNumberField(unittest.TestCase):
    def setUp(self):
        self.klass = PhoneNumberField
    #
#
run_test(TestPhoneNumberField)

class TestPositiveIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = PositiveIntegerField
    #
    def test_conversion(self):
        """ PositiveIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
    #

    def test_validate(self):
        """ PositiveIntegerField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, -1, {})
        cf.validate(1, {})
    #

    def test_conversion_blank(self):
        """ PositiveIntegerField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ PositiveIntegerField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestPositiveIntegerField)

class TestPositiveSmallIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = PositiveSmallIntegerField
    #

    def test_conversion(self):
        """ PositiveSmallIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
    #

    def test_validate(self):
        """ PositiveSmallIntegerField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, -1, {})
        cf.validate(1, {})
    #

    def test_conversion_blank(self):
        """ PositiveSmallIntegerField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ PositiveSmallIntegerField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestPositiveSmallIntegerField)

class TestSlugField(unittest.TestCase):
    def setUp(self):
        self.klass = SlugField
    #
    def test_validate(self):
        """ SlugField:: validate """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.validate, 'asAD d-ad_ad', {})
        cf.validate('valid-slug_', {})
    #

    def test_conversion_blank(self):
        """ SlugField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ SlugField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestSlugField)

class TestSmallIntegerField(unittest.TestCase):
    def setUp(self):
        self.klass = SmallIntegerField
    #

    def test_conversion(self):
        """ SmallIntegerField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('22'), 22)
    #

    def test_conversion_blank(self):
        """ SmallIntegerField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ SmallIntegerField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestSmallIntegerField)

class TestTextField(unittest.TestCase):
    def setUp(self):
        self.klass = TextField
    #

    def test_conversion_blank(self):
        """ TextField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ TextField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestTextField)

class TestTimeField(unittest.TestCase):
    def setUp(self):
        self.klass = TimeField
        self.data = '13:45:16'
        self.ret = time.strptime(self.data, '%H:%M:%S')
    #

    def test_conversion(self):
        """ TestTimeField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python(self.data), self.ret)
        # TODO: support for other date formats??
    #

    def test_conversion_null(self):
        """ TestTimeField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestTimeField)

class TestURLField(unittest.TestCase):
    def setUp(self):
        self.klass = URLField
    #
    
    def test_conversion(self):
        """ URLField:: conversion """
        cf = self.klass()
        self.assertEqual(cf.to_python('www.djangoproject.com'), 'www.djangoproject.com')
    #

    def test_validate(self):
        """ URLField:: conversion """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, 'this.not.exists')
        cf.to_python('www.djangoproject.com')
    #

    def test_conversion_blank(self):
        """ URLField:: conversion with blank value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, '')
        cf = self.klass(blank=True)
        self.assertEqual(cf.to_python(''), '')
    #

    def test_conversion_null(self):
        """ URLField:: conversion with NULL value """
        cf = self.klass()
        self.assertRaises(validators.ValidationError, cf.to_python, None)
        cf = self.klass(null=True)
        self.assertEqual(cf.to_python(None), None)
    #
#
run_test(TestURLField)
# TODO: USStateField
# XXX Maybe this is more for contrib than trunk?
class TestUSStateField(unittest.TestCase):
    def setUp(self):
        self.klass = USStateField
    #
#
run_test(TestUSStateField)
# TODO: XMLField
class TestXMLField(unittest.TestCase):
    def setUp(self):
        self.klass = XMLField
    #
#
run_test(TestXMLField)

# THE BIG TODO
# TODO: OrderingField
# TODO: ForeignKey
# TODO: OneToOneField
# TODO: ManyToManyField

if __name__ != '__main__':
    # for django runtests.py
    if errors or failures:
        assert False, 'some tests are failed (E: %d, F:%d)' % (errors, failures)
else:
    unittest.main()