"""
MX-specific Form helpers
"""

from django.forms.fields import RegexField


class MXRFCField(RegexField):
    """
    A field that accepts a 'classic' Mexican RFC (Registro Federal de Contribuyentes).
    
    More info about this: http://es.wikipedia.org/wiki/Registo_Federal_de_Contribuyentes_(M%C3%A9xico)
    """
    def __init__(self, *args, **kwargs):
        super(MXRFCField, self).__init__(r'^[a-zA-Z]{3}[a-zA-Z\d]?(\d\d)(0[\d]{1}|1[0-2]{1})(0[\d]{1}|[1-2]{1}\d|3[0-1])[a-zA-Z\d]{3}$', *args, **kwargs)

    def clean(self, value):
        value = super(MXRFCField, self).clean(value)
        return value.upper()


class MXCURPField(RegexField):
    """
    A field that accepts a 'classic' Mexican CURP (Clave Unica de Registro de Poblacion).
    
    More info about this: http://es.wikipedia.org/wiki/Clave_%C3%9Anica_de_Registro_de_Poblaci%C3%B3n_%28M%C3%A9xico%29
    """
    def __init__(self, *args, **kwargs):
        super(MXCURPField, self).__init__(r'^[a-zA-Z]{4}(\d\d)(0[\d]{1}|1[0-2]{1})(0[\d]{1}|[1-2]{1}\d|3[0-1])([hmHM]{1})[a-zA-Z]{5}\d{2}$', *args, **kwargs)

    def clean(self, value):
        value = super(MXCURPField, self).clean(value)
        return value.upper()


class MXZIPCodeField(RegexField):
    """
    A field that accepts a 'classic' Mexican ZIP Code.
    
    More info about this: http://es.wikipedia.org/wiki/C%C3%B3digo_postal_mexicano
    """
    def __init__(self, *args, **kwargs):
        super(MXZIPCodeField, self).__init__(r'^(\d){5}$', *args, **kwargs)



