"""
IT-specific Form helpers
"""

from django.core.validators import EMPTY_VALUES
from django.forms import ValidationError
from django.forms.fields import Field, RegexField, Select
from django.utils.translation import ugettext_lazy as _
from django.utils.encoding import smart_unicode
from django.contrib.localflavor.it.util import ssn_check_digit, vat_number_check_digit
import re

phone_digits_re = re.compile(r'^((\+|00)39\s)?[0-9]{2,4}\s[0-9]{5,8}$')

class ITZipCodeField(RegexField):
    default_error_messages = {
        'invalid': _('Enter a valid zip code.'),
    }
    def __init__(self, *args, **kwargs):
        super(ITZipCodeField, self).__init__(r'^\d{5}$',
        max_length=None, min_length=None, *args, **kwargs)

class ITRegionSelect(Select):
    """
    A Select widget that uses a list of IT regions as its choices.
    """
    def __init__(self, attrs=None):
        from it_region import REGION_CHOICES
        super(ITRegionSelect, self).__init__(attrs, choices=REGION_CHOICES)

class ITProvinceSelect(Select):
    """
    A Select widget that uses a list of IT provinces as its choices.
    """
    def __init__(self, attrs=None):
        from it_province import PROVINCE_CHOICES
        super(ITProvinceSelect, self).__init__(attrs, choices=PROVINCE_CHOICES)

class ITSocialSecurityNumberField(RegexField):
    """
    A form field that validates Italian Social Security numbers (codice fiscale).
    For reference see http://www.agenziaentrate.it/ and search for
    'Informazioni sulla codificazione delle persone fisiche'.
    """
    default_error_messages = {
        'invalid': _(u'Enter a valid Social Security number.'),
    }

    def __init__(self, *args, **kwargs):
        super(ITSocialSecurityNumberField, self).__init__(r'^\w{3}\s*\w{3}\s*\w{5}\s*\w{5}$',
        max_length=None, min_length=None, *args, **kwargs)

    def clean(self, value):
        value = super(ITSocialSecurityNumberField, self).clean(value)
        if value == u'':
            return value
        value = re.sub('\s', u'', value).upper()
        try:
            check_digit = ssn_check_digit(value)
        except ValueError:
            raise ValidationError(self.error_messages['invalid'])
        if not value[15] == check_digit:
            raise ValidationError(self.error_messages['invalid'])
        return value

class ITVatNumberField(Field):
    """
    A form field that validates Italian VAT numbers (partita IVA).
    """
    default_error_messages = {
        'invalid': _(u'Enter a valid VAT number.'),
    }

    def clean(self, value):
        value = super(ITVatNumberField, self).clean(value)
        if value == u'':
            return value
        try:
            vat_number = int(value)
        except ValueError:
            raise ValidationError(self.error_messages['invalid'])
        vat_number = str(vat_number).zfill(11)
        check_digit = vat_number_check_digit(vat_number[0:10])
        if not vat_number[10] == check_digit:
            raise ValidationError(self.error_messages['invalid'])
        return smart_unicode(vat_number)

class ITPhoneNumberField(Field):
    """
    A form field that validates Italian Phone numbers (fixed and mobile)
    The correct format is XX(XX) XXXXX(XXX). Italian prefixes are between 2 
    and 4 digits long for fixed line and 3 digits for all mobile operators.
    Over all the number is between 10 and 11 digits long.
    '0039 XX(XX) XXXXX(XXX)' and '+39 XX(XX) XXXXX(XXX)' validate too, but the 
    international prefix is stripped from the number.
    Spaces between prefixes and number are mandatory.
    """
    default_error_messages = {
        'invalid':_(u'Enter a valid phone number'),
    }

    def clean(self, value):

        value = super(ITPhoneNumberField, self).clean(value)
        if value == u'':
            return value
       
        n = phone_digits_re.search(value)
        
        if not n:
            raise ValidationError(self.error_messages['invalid'])
        
        # Import a list of valid prefixes
        from it_phones import PHONE_CHOICES, MOBILE_CHOICES
             
        # Split the number and, if present, strip the international prefix from the number
        m = value.split(" ")
        
        if (m[0] == '+39') or (m[0] == '0039'):
            del m[0]
        
        # Check if the number is no longer than 11 digits
        if (len(m[0]) + len(m[1])) > 11:
            raise ValidationError(self.error_messages['invalid'])

        # Check the prefix against a list of valid fixed and mobile prefixes    
        if m[0] in PHONE_CHOICES or m[0] in MOBILE_CHOICES:
            return u'%s %s' % (m[0], m[1])
        
        raise ValidationError(self.error_messages['invalid'])
